<?php

namespace App\Controller;

use App\Controller\Base\AbstractCreditoController;
use App\Electronico\Comprobante;
use App\Entity\Model\Customer;
use App\Entity\Model\Empresa;
use App\Entity\Model\Ncredito;
use App\Entity\Model\Item;
use App\Form\CreditoType;
use App\Form\CustomerType;
use App\Service\EmFactory;
use App\Util\Funciones;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @Route("/credito")
 */
class CreditoController extends AbstractCreditoController
{

    public $logger;
    public $translator;

    /**
     * @Route("", name="credito_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $emisor = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(\App\Entity\Model\Ncredito::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 20;

        $form = $this->createForm(\App\Form\SearchGenericType::class, null, [
            'action' => $this->generateUrl('credito_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearch($form->getData(), $limit, $request->query->getInt('page', 1), $emisor->getId());
        } else {
            $pagination = $repo->paginatedSearch([], $limit, $request->query->getInt('page', 1), $emisor->getId());
        }

        $creditos = [];
        foreach ($pagination->getItems() as $item) {
            $creditos[] = $item;
        }

        $listForm = $this->createForm(\App\Form\ListGenericType::class, $creditos, [
            'action' => $this->generateUrl('credito_index'),
        ]);
        $listForm->handleRequest($request);
        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if (empty($data['creditos'])) {
                $this->addTranslatedMessage('flash.nothing_selected', 'warning');
            } elseif ($request->request->has('delete')) {
                return $this->bulkDelete($data['creditos']);
            } elseif ($request->request->has('pdf')) {
                return $this->bulkPdf($data['creditos']);
            } elseif ($request->request->has('print')) {
                return $this->bulkPrint($data['creditos']);
            } elseif ($request->request->has('email')) {
                return $this->bulkEmail($data['creditos']);
            }
        }

        return $this->render('Credito\index.html.twig',
            [
                'creditos' => $pagination,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
                'search_form' => $form->createView(),
                'list_form' => $listForm->createView(),
            ]);
    }


    /**
     * @Route("/show/{slug}ncr{id}", name="credito_show")
     *
     */
    public function showAction($id, EmFactory $emFactory, $slug, TranslatorInterface $translator, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $emisor = $empresaRepo->findOneByUser($user->getRuc());

        $entity = $em->getRepository(\App\Entity\Model\Ncredito::class)->findBySlug($slug, $id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Ncredito entity.');
        }

        $defaultData = ['slug' => $slug, 'id'=>$id, 'customerEmail' => $entity->getCustomerEmail()];

        $form = $this->createFormBuilder($defaultData)
            ->add('slug', HiddenType::class)
            ->add('id', HiddenType::class)
            ->add('customerEmail')
            ->setAction($this->generateUrl('credito_show', ['id' => $id, 'slug'=>$slug]))
            ->getForm();

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // data is an array with "name", "email", and "message" keys
            $data = $form->getData();

            if ($request->request->has('Form-pdf')) {
                $this->generarPdf($entity);
            } elseif ($request->request->has('Form-email')) {
                if (isset($data['customerEmail'])) {
                    $email = $data['customerEmail'];
                    $this->enviarMail($email, $entity, $em);
                } else
                    $this->addTranslatedMessage('Email del cliente nulo o en blanco', 'warning');
            } elseif ($request->request->has('Form-edit')) {
                if ($entity->getAutorizado() === false)
                    return $this->redirectToRoute('credito_edit', ['id' => $entity->getId(), 'slug' => $slug]);
            } elseif ($request->request->has('Form-delete')) {
                $error = $this->delete($entity, $em);
                if ($error)
                    return $this->redirectToRoute('credito_index');
            } elseif ($request->request->has('Form-anular')) {
                $entity->setAnulado(true);
                $em->persist($entity);
                $em->flush();
            } elseif ($request->request->has('Form-enviar')) {
                $this->enviarSriOnline($entity, $em);
            } elseif ($request->request->has('Form-auto')) {
                if ($entity->getAutorizado() === false) {
                    $resp = $this->consultarAutorizacion($entity, $em);
                    if ($resp != null)
                        $this->addTranslatedMessage($resp, 'danger');
                }

            }
        }

        if ($entity->getMensajeError()) {
            $this->addTranslatedMessage($entity->getMensajeError(), 'danger');
        }

        /*if (!$entity->isClosed()) {
            // When the credito is open send to the edit form by default.
            return $this->redirect($this->generateUrl('credito_edit', array('id' => $id)));
        }
        */

        return $this->render('Credito\show.html.twig',
            [
                'entity' => $entity,
                'form' => $form->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
                'decpunit' => $emisor->getDecPunit(),
            ]);
    }

    /**
     * @Route("/new", name="credito_add")
     *
     */
    public function newAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $msg='';
        $puedefacturar = Funciones::getValidaPuedeFacturar($empresa, $em, $msg);
        if ($empresa->getPuedefacturar() === false && $empresa->getTipoAmbiente() === "2") {
            $this->addTranslatedMessage($msg, 'danger');
            return $this->redirect($this->generateUrl('credito_index'));
        }

        $empresa = Funciones::getValidaEmitidos($empresa, $em);
        if ($empresa->getPuedefacturar() === false && $empresa->getTipoAmbiente() === "2") {
            $this->addTranslatedMessage('Número de comprobantes emitidos excede del plan contratado ', 'danger');
            return $this->redirect($this->generateUrl('credito_index'));
        }

        $credito = new Ncredito();
        $newItem = new Item();

        $tax = $em->getRepository(\App\Entity\Model\Tax::class)->findTaxDefault($empresa->getId());
        $newItem->setTaxes($tax);
        $credito->addItem($newItem);

        $form = $this->createForm(\App\Form\CreditoType::class, $credito, [
            'action' => $this->generateUrl('credito_add'),
        ]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            if ($request->request->has('save_draft')) {
                $credito->setStatus(Ncredito::DRAFT);
            } else {
                // Any save action transforms this to opened.
                $credito->setStatus(Ncredito::OPENED);
            }

            /*if ($credito->getCustomer() == null) {
                $this->addTranslatedMessage('Cliente no se encuentra registrado', 'danger');
            } else {*/
                $serie = $credito->getSeries()->getValue();
                $credito->setSerie($serie);
                $this->cargarImpuestos($credito);
                $credito->setEmpresa($empresa);
                $credito->setUsuario($user->getId());
                $credito->setAmbiente($empresa->getTipoAmbiente());

                $em->persist($credito);
                $em->flush();
                //$this->addTranslatedMessage('flash.added');

                $credito_id = $credito->getId();

                $credito = $this->generarXml($credito, $em);

                if ($empresa->getEnvioAutomatico()) {

                   /* if ($credito !== null) {
                        $app_url = $this->getParameter('cola_url');

                        $error = false;

                        $resp = RestApiFunciones::enviarComprobanteCola($error,
                            $credito->getCustomerName(),
                            $credito->getCustomerEmail(),
                            $user->getId(),
                            $credito->getXml(),
                            $credito->getClaveAcceso(),
                            $app_url,
                            'ncredito'
                        );

                        if ($error) {
                            $this->addTranslatedMessage($resp, 'danger');
                            $credito->setMensajeError($resp);
                        }

                        $em->persist($credito);
                        $em->flush();

                    }

                    return $this->redirect($this->generateUrl('credito_index'));
                   */

                }

                return $this->redirectToRoute('credito_show', ['id' => $credito_id, 'slug' => $credito->getSlug()]);
            //}
        }

        $customer = new Customer();

        $formcustomer = $this->createForm(\App\Form\CustomerType::class, $customer, [
            'action' => $this->generateUrl('rest_customer_add'),
        ]);
        $formcustomer->handleRequest($request);

        return $this->render('Credito\edit.html.twig',
            [
                'form' => $form->createView(),
                'formcustomer' => $formcustomer->createView(),
                'customer' => $customer,
                'entity' => $credito,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
            ]);
    }


    /**
     * @Route("/edit/{slug}ncr{id}", name="credito_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $entity = $em->getRepository(Ncredito::class)->findBySlug($slug, $id);
        if (!$entity && $entity->getEmpresa() != $empresa) {
            throw $this->createNotFoundException('Unable to find Ncredito entity.');
        }

        if($entity->getAutorizado()){
            return $this->redirectToRoute('credito_show', ['id' => $id, 'slug' => $slug]);
        }

        $form = $this->createForm(CreditoType::class, $entity, [
            'action' => $this->generateUrl('credito_edit', ['id' => $id, 'slug' => $slug]),
        ]);
        $form->handleRequest($request);

        $entity->setEmpresa($empresa);

        if ($form->isSubmitted() && $form->isValid()) {
            $redirectRoute = 'credito_show';

            $serie = $entity->getSeries()->getValue();
            $entity->setSerie($serie);
            $entity->setUsuario($user->getId());
            $this->cargarImpuestos($entity);

            if ($request->request->has('save_draft')) {
                $entity->setStatus(Ncredito::DRAFT);
            } elseif ($request->request->has('save_close')) {
                $entity->setForcefullyClosed(true);
            } elseif ($entity->isDraft()) {
                // Any save action transforms this to opened.
                $entity->setStatus(Ncredito::OPENED);
            }

            /*
            // See if one of PDF/Print buttons was clicked.
            if ($request->request->has('save_pdf')) {
                $redirectRoute = 'credito_show_pdf';
            } elseif ($request->request->has('save_print')) {
                $this->get('session')->set('credito_auto_print', $id);
            }

            */

            // Save.
            $em->persist($entity);
            $em->flush();
            //$this->addTranslatedMessage('flash.updated');

            $entity = $this->generarXml($entity, $em);

            return $this->redirectToRoute($redirectRoute, ['id' => $id, 'slug' => $slug]);
        }

        $customer = new Customer();

        $formcustomer = $this->createForm(CustomerType::class, $customer, [
            'action' => $this->generateUrl('rest_customer_add'),
        ]);
        $formcustomer->handleRequest($request);

        return $this->render('Credito\edit.html.twig',
            [
                'entity' => $entity,
                'form' => $form->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'formcustomer' => $formcustomer->createView(),
            ]);
    }

    public function enviarSriOnline(Ncredito $ncredito, EntityManagerInterface $em)
    {
        $ncredito = $this->generarXml($ncredito, $em);

        if($ncredito !== null) {
            $app_url = $this->getParameter('api_url');

            $error = false;
            $resp = RestApiFunciones::enviarComprobante($error, $ncredito->getXml(), $ncredito->getClaveAcceso(), $app_url);

            if ($error) {
                $this->addTranslatedMessage($resp->message, 'danger');
            } elseif ($resp->estado === 'recibido') {
                $resp = $this->consultarAutorizacion($ncredito, $em);
                if ($resp !== null)
                    $this->addTranslatedMessage($resp, 'danger');
            } else {
                $ncredito->setMensajeError($resp->message);
                $this->addTranslatedMessage($resp->message, 'danger');
                $em->persist($ncredito);
                $em->flush();
            }

        }
    }

    /**
     * @Route("/payments/{slug}ncr{id}", name="credito_payments")
     *
     */
    public function paymentsAction(Request $request, $slug, EmFactory $emFactory, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $emisor = null;
        $this->translator = $translator;

        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        // Return all payments
        $credito = $em->getRepository(\App\Entity\Model\Ncredito::class)->findBySlug($slug);
        if (!$credito) {
            throw $this->createNotFoundException('Unable to find Ncredito entity.');
        }

        $payment = new Payment;
        $addForm = $this->createForm(\App\Form\PaymentType::class, $payment, [
            'action' => $this->generateUrl('credito_payments', ['id' => $credito->getId(), 'slug' => $slug]),
        ]);
        $addForm->handleRequest($request);
        if ($addForm->isSubmitted() && $addForm->isValid()) {
            $credito->addPayment($payment);
            $em->persist($credito);
            $em->flush();
            $this->addTranslatedMessage('payment.flash.added');

            // Rebuild the query, since we have new objects now.
            return $this->redirectToRoute('credito_index');
        }

        $listForm = $this->createForm('App\Form\ListNcreditoPaymentType', $credito->getPayments()->getValues(), [
            'action' => $this->generateUrl('credito_payments', ['id' => $credito->getId(), 'slug' => $slug]),
        ]);
        $listForm->handleRequest($request);

        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            foreach ($data['payments'] as $payment) {
                $credito->removePayment($payment);
                $em->persist($credito);
                $em->flush();
            }

            $this->addTranslatedMessage('payment.flash.bulk_deleted');

            // Rebuild the query, since some objects are now missing.
            return $this->redirectToRoute('credito_index');
        }

        return $this->render('Payment\list.html.twig',
            [
                'creditoId' => $creditoId,
                'add_form' => $addForm->createView(),
                'list_form' => $listForm->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor === null ? 'USD' : $emisor->getCurrency(),
            ]);
    }

    /**
     * @Route("/form-totals", name="credito_form_totals")
     */
    public function getNcreditoFormTotals(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();

        $post = $request->request->get('credito');
        if (!$post) {
            throw new NotFoundHttpException;
        }

        $response = $this->getNcreditoTotalsFromPost($post, new Ncredito, $request->getLocale(), $em);

        return new JsonResponse($response);
    }

    /**
     * @Route("/pdfpreview/{slug}inv{id}", name="credito_show_pdf_preview")
     *
     */
    public function showOnlinePdfAction($id,  $slug, EmFactory $emFactory, TranslatorInterface $translator, Request $request, LoggerInterface $logger): \Symfony\Component\HttpFoundation\Response
    {
        $this->logger = $logger;
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoice = $em->getRepository(\App\Entity\Model\Ncredito::class)->findBySlug($slug, $id);
        if (!$invoice) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        $filename = 'NC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $xml = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url').'ncreditoride.php';

        $error = false;
        $mensaje = "";
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            file_put_contents($filename, $docPdf);

            $pdf = base64_encode(file_get_contents($filename));

            $response = new Response($pdf);
            $response->headers->set('Content-Type', 'application/octet-stream');
            $response->headers->set('Content-Description', 'File Transfer');
            $response->headers->set('Content-Disposition', 'attachment; filename="'.$filename.'"');
            // $response->headers->set('Expires', '0');
            // $response->headers->set('Content-Transfer-Encoding', 'binary');
            $response->headers->set('Content-length', strlen($pdf));
            $response->headers->set('Cache-Control', 'no-cache private');
            // $response->headers->set('Pragma', 'public');
            // Send headers before outputting anything
            $response->sendHeaders();



            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }

            return $response;
        }
    }
}
