<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Sluggable\Util\Urlizer;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * App\Entity\Model\Reembolso
 *
 * @ORM\Entity(repositoryClass="App\Repository\ItemRepository")
 * @ORM\Table(indexes={
 *    @ORM\Index(name="liquidacion_item_desc_idx", columns={"provreembolso_identification"})
 * })
 */
class Reembolso
{

    /**
     *
     * @ORM\Column(type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private  $id = null;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Provider", inversedBy="proveedores")
     * @ORM\JoinColumn(name="provider_id", referencedColumnName="id")
     */
    protected $provider;

    /**
     *
     * @ORM\Column(name="provreembolso_name", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el nombre del proveedor")
     */
    private  $provreembolso_name = null;


    /**
     *
     * @ORM\Column(name="provreembolso_identification", type="string", length=128, nullable=true)
     * @Assert\NotBlank(message = "Ingrese el número de identificación del proveedor")
     */
    private $provreembolso_identification = null;

    /**
     * @ORM\Column(name="provider_codpais", type="string", length=3)
     */
    private  $provider_codpais = '593';

    /**
     * @ORM\Column(name="serie_doc", type="string", length=6)
     */
    private  $serieDocumento = null;

    /**
     * @ORM\Column(name="secuencial_doc", type="string", length=9)
     */
    private  $secuencialDoc = null;

    /**
     * @var date $emisionDocumento
     *
     * @ORM\Column(name="fecemision_doc", type="date", nullable=true)
     * @Assert\Date()
     */
    private $emisionDocumento;

    /**
     * @ORM\Column(name="auto_doc", type="string", length=49)
     */
    private  $autoDocumento = null;

    /**
     * @var
     *
     * @ORM\Column(name="imponible", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $imponible;

    /**
     * @var
     *
     * @ORM\Column(name="baseiva", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $baseIva;


    /**
     * @var
     *
     * @ORM\Column(name="basecero", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $baseCero;

    /**
     * @var
     *
     * @ORM\Column(type="decimal", scale=2, precision=15, nullable=true)
     */
    private $porIva;

    /**
     * @var
     *
     * @ORM\Column(type="decimal", precision=15, scale=2)
     * @Assert\NotBlank()
     */
    private $valiva;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\TipoDocReembolso", inversedBy="itemsdocreembolsos")
     * @ORM\JoinColumn(name="tipodocreembolso_id", referencedColumnName="id", onDelete="SET NULL")
     */
    private $tipoDocReembolso;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Liquidacion", inversedBy="reembolsos")
     **/
    private $liquidacion;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Tax")
     * @ORM\JoinColumn(name="tax_id", referencedColumnName="id", onDelete="SET NULL")
     *
     */
    private $taxes;


    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @return mixed
     */
    public function getTipoDocReembolso()
    {
        return $this->tipoDocReembolso;
    }

    /**
     * @param mixed $tipoDocReembolso
     */
    public function setTipoDocReembolso($tipoDocReembolso): void
    {
        $this->tipoDocReembolso = $tipoDocReembolso;
    }

    /**
     * @return mixed
     */
    public function getProvider()
    {
        return $this->provider;
    }

    /**
     * @param mixed $provider
     */
    public function setProvider($provider): void
    {
        $this->provider = $provider;
    }

    /**
     * @return string
     */
    public function getProvreembolsoName()
    {
        return $this->provreembolso_name;
    }

    /**
     * @param string $provreembolso_name
     */
    public function setProvreembolsoName($provreembolso_name): void
    {
        $this->provreembolso_name = $provreembolso_name;
    }

    /**
     * @return string
     */
    public function getProvreembolsoIdentification()
    {
        return $this->provreembolso_identification;
    }

    /**
     * @param string $provreembolso_identification
     */
    public function setProvreembolsoIdentification(string $provreembolso_identification): void
    {
        $this->provreembolso_identification = $provreembolso_identification;
    }

    /**
     * @return string
     */
    public function getSerieDocumento()
    {
        return $this->serieDocumento;
    }

    /**
     * @param string $serieDocumento
     */
    public function setSerieDocumento($serieDocumento): void
    {
        $this->serieDocumento = $serieDocumento;
    }


    /**
     * @return string
     */
    public function getSecuencialDoc()
    {
        return $this->secuencialDoc;
    }

    /**
     * @param string $secuencialDoc
     */
    public function setSecuencialDoc($secuencialDoc): void
    {
        $this->secuencialDoc = $secuencialDoc;
    }

    /**
     * @return date
     */
    public function getEmisionDocumento()
    {
        return $this->emisionDocumento;
    }

    /**
     * @param date $emisionDocumento
     */
    public function setEmisionDocumento($emisionDocumento): void
    {
        $this->emisionDocumento = $emisionDocumento instanceof \DateTime ?
            $emisionDocumento: new \DateTime($emisionDocumento);
    }

    /**
     * @return string
     */
    public function getAutoDocumento()
    {
        return $this->autoDocumento;
    }

    /**
     * @param string $autoDocumento
     */
    public function setAutoDocumento($autoDocumento): void
    {
        $this->autoDocumento = $autoDocumento;
    }

    /**
     * @return string
     */
    public function getBaseIva()
    {
        return $this->baseIva;
    }

    /**
     * @param string $baseIva
     */
    public function setBaseIva($baseIva): void
    {
        $this->baseIva = $baseIva;
    }

    /**
     * @return string
     */
    public function getBaseCero()
    {
        return $this->baseCero;
    }

    /**
     * @param string $baseCero
     */
    public function setBaseCero($baseCero): void
    {
        $this->baseCero = $baseCero;
    }

    /**
     * @return mixed
     */
    public function getPorIva()
    {
        return $this->porIva;
    }

    /**
     * @param mixed $porIva
     */
    public function setPorIva($porIva): void
    {
        $this->porIva = $porIva;
    }

    /**
     * @return decimal
     */
    public function getValiva()
    {
        return $this->valiva;
    }

    /**
     * @param decimal $valiva
     */
    public function setValiva($valiva): void
    {
        $this->valiva = $valiva;
    }

    /**
     * @return mixed
     */
    public function getLiquidacion()
    {
        return $this->liquidacion;
    }

    /**
     * @param mixed $liquidacion
     */
    public function setLiquidacion($liquidacion): void
    {
        $this->liquidacion = $liquidacion;
    }

    /**
     * @return string
     */
    public function getImponible()
    {
        return $this->imponible;
    }

    /**
     * @param string $imponible
     */
    public function setImponible($imponible): void
    {
        $this->imponible = $imponible;
    }

    /**
     * Add taxes
     *
     * @param App\Entity\Model\Tax $tax
     */
    public function setTaxes(Tax $tax)
    {
        $this->taxes = $tax;
    }


    /**
     * Get taxes
     *
     * @return Tax
     */
    public function getTaxes()
    {
        return $this->taxes;
    }

    /**
     * @return string
     */
    public function getProviderCodpais(): string
    {
        return $this->provider_codpais;
    }


    public function numeroSecuencial()
    {
        return sprintf("%09d",$this->getSecuencialDoc());
    }

    /**
     *
     *
     * Remove Tax
     *
     * @param App\Entity\Model\Tax
     */
    /*public function removeTax(\App\Entity\Model\Tax $tax)
    {
        $this->taxes->removeElement($tax);
    }

    /** **************** CUSTOM METHODS ************* */

    /**
     * Get taxes percent
     *
     * @param tax_names array if present shows only percent of those taxes
     * @return integer total percent of taxes to apply
     */
    public function getTaxesPercent($tax_names = [])
    {
        $tax_names = is_array($tax_names) ?
            array_map([\Gedmo\Sluggable\Util\Urlizer::class, 'urlize'], $tax_names):
            [Urlizer::urlize($tax_names)] ;
        $total = 0;
        //foreach ($this->getTaxes() as $tax) {
        if($this->taxes != null){
            $tax = $this->taxes;
            if (count($tax_names)==0 ||
                in_array(Urlizer::urlize(str_replace(' ', '', $tax->getName())), $tax_names)) {
                $total += $tax->getValue();
            }
        }

        return $total;
    }



    /**
     * Try to capture a "getTaxAmountTAXNAME" method.
     * This is to be able to use 'liquidacion.tax_amount_TAXNAME' in the templates
     *
     * @author JoeZ99 <jzarate@gmail.com>
     */
    public function __call($data, $argument)
    {
        if (strpos($data, 'getTaxAmount') === 0 && strlen($data)>12) {
            $tax_name = substr($data, 12);
            return $this->getTaxAmount($tax_name);
        }

        return false;
    }

    /**
     * Again, capture hipothetical {{liquidacion.base_amount}} and the like
     *
     */
    public function __get($name)
    {
        $method = Inflector::camelize(sprintf('get_%s', $name));
        if (method_exists($this, $method)) {
            return $this->$method();
        }

        return false;
    }

    /**
     * Twig template system needs this to answer true for the specified properties
     */
    public function __isset($name)
    {
        $classVars = array_keys(get_object_vars($this));
        $extraVars = ['base_amount', 'porcentaje_amount', 'gross_amount'];
        return in_array($name, array_merge($classVars, $extraVars));
    }

    public function __toString()
    {
        return (string) $this->provreembolso_name.': '.$this->provreembolso_identification;
    }

    public function __construct()
    {
    }

    public function jsonSerialize()
    {
        return [
            'id' => $this->getId(),
            'subtotal' => $this->getSubtotal(),
            'porcentaje' => $this->getPorIva(),
        ];
    }

    /**
     * Get base amount
     *
     * @return float gross price of the item (times quantity)
     */
    public function getBaseAmount()
    {
        return $this->imponible;
    }

    /**
     * Get net amount
     *
     * @return float price with discount
     */
    public function getNetAmount()
    {
        return $this->getBaseAmount() ;//- $this->getDiscountAmount();
    }

    /**
     * Get tax amount
     *
     * @param array tax_names. if present, returns the amount for those taxes
     * @return float amount to tax
     */
    public function getTaxAmount($tax_names = [])
    {
        return $this->getNetAmount() * $this->getTaxesPercent($tax_names) / 100;
    }
}
