<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * App\Entity\Model\Retencion
 *
 * @ORM\Table(indexes={
 *    @ORM\Index(name="retencion_cstnm_idx", columns={"provider_name"}),
 *    @ORM\Index(name="retencion_cstid_idx", columns={"provider_identification"}),
 *    @ORM\Index(name="retencion_cstml_idx", columns={"provider_email"})
 * })
 * @ORM\Entity(repositoryClass="App\Repository\RetencionRepository")
 * @ORM\HasLifecycleCallbacks()
 */
class Retencion extends AbstractRetencion
{
    /**
     * @ORM\Column(name="sent_by_email", type="boolean", nullable=true)
     */
    private  $sent_by_email = null;

    /**
     * @ORM\Column(type="string", length=6)
     */
    private  $serie = '';

    /**
     * @ORM\Column(name="number", type="integer", nullable=true)
     */
    private  $number = null;

    /**
     * @var date $issue_date
     *
     * @ORM\Column(name="issue_date", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $issue_date;

    /**
     * @var string
     * @ORM\Column(type="string", length=9)
     */
    private $numero_compra;

    /**
     * @ORM\Column(type="string", length=6)
     * @Assert\Length(min=6, max=6)
     */
    private $serie_compra = null;

    /**
     * @var date $fecha_compra
     *
     * @ORM\Column(name="fecha_compra", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $fecha_compra;

    /**
     * @ORM\ManyToMany(targetEntity="App\Entity\Model\ItemRetencion", cascade={"all"}, inversedBy="retenciones")
     * @ORM\JoinTable(name="retencion_items",
     *      joinColumns={@ORM\JoinColumn(name="retencion_id", referencedColumnName="id", onDelete="CASCADE")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="item_id", referencedColumnName="id", unique=true, onDelete="CASCADE")}
     * )
     * @Assert\NotBlank()
     */
    public $items;

    /**
     * @ORM\Column(name="closed", type="boolean", nullable=true)
     */
    private  $forcefully_closed = true;

    /**
     * @ORM\Column(name="baserenta", type="decimal", scale=2, precision=15, nullable=true)
     */
    private  $baserenta = null;

    /**
     * @ORM\Column(name="baseiva", type="decimal", scale=2, precision=15, nullable=true)
     */
    private  $baseiva = null;

    /**
     * @ORM\Column(name="mes", type="string", length=2, nullable=true)
     */
    private  $mes = null;

    /**
     * @ORM\Column(name="anio", type="string", length=4, nullable=true)
     */
    private  $anio = null;

    /**
     * @ORM\Column(name="claveacceso", type="string", length=49)
     */
    private  $claveAcceso = '';

    /**
     * @ORM\Column( type="boolean")
     */
    private  $autorizado = false;

    /**
     * @ORM\Column( type="boolean")
     */
    private  $anulado = false;


    /**
     * @ORM\Column( type="string", length=49, nullable=true)
     */
    private  $autorizacion = null;

    /**
     * @ORM\Column(type="string", length=50, nullable=true)
     */
    private $fecha_autorizacion = null;

    /**
     * @ORM\Column(type="string", length=2, nullable=true)
     */
    private  $codigoDocSustento = null;


    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $xml;

    /**
     * @ORM\Column( type="integer", length=200, nullable=true)
     */
    protected $ambiente;

    /**
     * @ORM\Column( type="boolean")
     */
    private  $sinrespuesta = false;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $mensajeError;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $xmlAutorizado;

    /**
     * @ORM\Column(type="blob", nullable=true)
     */
    private $docpdf;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Empresa")
     * @ORM\JoinColumn(name="empresa_id", referencedColumnName="id", onDelete="SET NULL")
     */
    public $empresa;


    public function __construct()
    {
        parent::__construct();
        $this->issue_date = new \DateTime();
        $this->fecha_compra = new \DateTime();

    }

    /**
     * @return boolean
     */
    public function isClosed()
    {
        return $this->status === Retencion::CLOSED;
    }

    /**
     * @return boolean
     */
    public function isOpen()
    {
        return in_array($this->status, [Retencion::OPENED, Retencion::OVERDUE], true);
    }

    /**
     * @return boolean
     */
    public function isOverdue()
    {
        return $this->status === Retencion::OVERDUE;
    }

    /**
     * @return boolean
     */
    public function isDraft()
    {
        return $this->status === Retencion::DRAFT;
    }

    /**
     * Set sent_by_email
     *
     * @param boolean $sentByEmail
     */
    public function setSentByEmail($sentByEmail)
    {
        $this->sent_by_email = $sentByEmail;
    }

    /**
     * Get sent_by_email
     *
     * @return boolean
     */
    public function isSentByEmail(): bool
    {
        return (bool) $this->sent_by_email;
    }

    /**
     * Set number
     *
     * @param integer $number
     */
    public function setNumber($number)
    {
        $this->number = $number;
    }

    /**
     * Get number
     *
     * @return integer
     */
    public function getNumber()
    {
        return $this->number;
    }

    /**
     * Set issue_date
     *
     * @param date $issueDate
     */
    public function setIssueDate($issueDate)
    {
        $this->issue_date = $issueDate instanceof \DateTime ?
            $issueDate: new \DateTime($issueDate);
    }

    /**
     * Get issue_date
     *
     * @return date
     */
    public function getIssueDate()
    {
        return $this->issue_date;
    }

    /**
     * Set fecha_compra
     *
     * @param date $fechaCompra
     */
    public function setFechaCompra($fechaCompra)
    {
        $this->fecha_compra = $fechaCompra instanceof \DateTime ?
            $fechaCompra : new \DateTime($fechaCompra);
    }

    /**
     * Get fecha_compra
     *
     * @return date
     */
    public function getFechaCompra()
    {
        return $this->fecha_compra;
    }

    /**
     * @return string
     */
    public function getSerieCompra()
    {
        return $this->serie_compra;
    }

    /**
     * @param string $serie_compra
     */
    public function setSerieCompra(string $serie_compra)
    {
        $this->serie_compra = $serie_compra;
    }


    public function setForcefullyClosed($value)
    {
        $this->forcefully_closed = $value;
    }

    public function isForcefullyClosed()
    {
        return $this->forcefully_closed;
    }



    /** **************** CUSTOM METHODS AND PROPERTIES **************  */

    /**
     * TODO: provide the serie .
     */
    public function __toString()
    {
        return $this->label();
    }

    public function label(string $draftLabel = '[draft]')
    {
        /*$label = '';
        $label .= $this->serie.'-';
        $label .= $this->isDraft() ? $draftLabel : str_pad($this->getNumber(), 9, '0', STR_PAD_LEFT);;

        return $label;*/

        $label = sprintf("%09d",$this->getNumber());
        return $label;
    }

    /**
     * @var int
     */
    public const DRAFT    = 0;

    /**
     * @var int
     */
    public const CLOSED   = 1;

    /**
     * @var int
     */
    public const OPENED   = 2;

    /**
     * @var int
     */
    public const OVERDUE  = 3;

    public function getDueAmount()
    {
        if ($this->isDraft()) {
            return null;
        }

        return $this->getGrossAmount();
    }

    /**
     * try to catch custom methods to be used in twig templates
     */
    public function __get($name)
    {
        if (strpos($name, 'tax_amount_') === 0) {
            return $this->calculate($name, true);
        }

        $method = Inflector::camelize(sprintf('get_%s', $name));
        if (method_exists($this, $method)) {
            return $this->$method();
        }

        return false;
    }

    public function __isset($name)
    {
        if (strpos($name, 'tax_amount_') === 0) {
            return true;
        }

        if ($name == 'due_amount') {
            return true;
        }

        if (array_key_exists($name, get_object_vars($this))) {
            return true;
        }

        return parent::__isset($name);
    }

    public function getStatusString()
    {
        switch ($this->status) {
            case Retencion::DRAFT;
                $status = 'draft';
                break;
            case Retencion::CLOSED;
                $status = 'closed';
                break;
            case Retencion::OPENED;
                $status = 'opened';
                break;
            case Retencion::OVERDUE:
                $status = 'overdue';
                break;
            default:
                $status = 'unknown';
                break;
        }

        return $status;
    }

    public function getStatusAutorizado()
    {
        if ($this->anulado) {
            $status = 'overdue';
        } elseif ($this->autorizado) {
            $status = 'closed';
        } else
            $status = 'opened';

        return $status;
    }

    /**
     * checkStatus
     * checks and sets the status
     *
     * @return App\Retencion $this
     */
    public function checkStatus()
    {
        if ($this->status == Retencion::DRAFT) {
            return $this;
        }

        if ($this->isForcefullyClosed() || $this->getDueAmount() == 0) {
            $this->setStatus(Retencion::CLOSED);
        } elseif ($this->getDueDate()->getTimestamp() > strtotime(date('Y-m-d'))) {
            $this->setStatus(Retencion::OPENED);
        } else {
            $this->setStatus(Retencion::OVERDUE);
        }

        return $this;
    }

    public function checkAmounts()
    {
        parent::checkAmounts();

        return $this;
    }

    public function checkNumber(LifecycleEventArgs $args)
    {
        // compute the number of retencion
        if ((!$this->number && $this->status != self::DRAFT) ||
            ($args instanceof PreUpdateEventArgs && $args->hasChangedField('series') && $this->status != self::DRAFT)
        ) {
            $repo = $args->getObjectManager()->getRepository(Retencion::class);
            $series = $this->getSeries();
            if ($repo && $series && $this->getNumber() <= 0) {

                $this->setNumber($repo->getNextNumber($series, $this->getEmpresa()->getId()));
            }
        }
    }


    /* ********** LIFECYCLE CALLBACKS *********** */

    /**
     * @ORM\PrePersist
     * @ORM\PreUpdate
     */
    public function preSave(LifecycleEventArgs $args)
    {
        $this->setUpdatedAt(new \DateTime('now'));

        $this->checkAmounts();
        parent::presave($args);

        if ($this->getCreatedAt() == null) {
            $this->setCreatedAt(new \DateTime('now'));
            $this->checkNumber($args);
        }

    }

    /**
     * @return decimal
     */
    public function getBaseRenta()
    {
        return $this->baserenta;
    }

    /**
     * @param decimal $baserenta
     */
    public function setBaseRenta($baserenta)
    {
        $this->baserenta = $baserenta;
    }

    /**
     * @return decimal
     */
    public function getBaseIva()
    {
        return $this->baseiva;
    }

    /**
     * @param decimal $baseiva
     */
    public function setBaseIva($baseiva)
    {
        $this->baseiva = $baseiva;
    }

    /**
     * @return string
     */
    public function getMes()
    {
        return $this->mes;
    }

    /**
     * @return string
     */
    public function getNombreMes()
    {
        switch ($this->mes){
            case '01':
                return 'Enero';
            case '02':
                return 'Febrero';
            case '03':
                return 'Marzo';
            case '04':
                return 'Abril';
            case '05':
                return 'Mayo';
            case '06':
                return 'Junio';
            case '07':
                return 'Julio';
            case '08':
                return 'Agosto';
            case '09':
                return 'Septiembre';
            case '10':
                return 'Octubre';
            case '11':
                return 'Noviembre';
            case '12':
                return 'Diciembre';



        }

        return '';
    }

    /**
     * @param string $mes
     */
    public function setMes(string $mes)
    {
        $this->mes = $mes;
    }

    /**
     * @return string
     */
    public function getClaveAcceso()
    {
        return $this->claveAcceso;
    }

    /**
     * @param string $claveAcceso
     */
    public function setClaveAcceso($claveAcceso)
    {
        $this->claveAcceso = $claveAcceso;
    }

    /**
     * @return string
     */
    public function getSerie()
    {
        return $this->serie;
    }

    /**
     * @param string $serie
     */
    public function setSerie($serie)
    {
        $this->serie = $serie;
    }

    /**
     * @return bool
     */
    public function getAutorizado()
    {
        return $this->autorizado;
    }

    /**
     * @param bool $autorizado
     */
    public function setAutorizado($autorizado)
    {
        $this->autorizado = $autorizado;
    }

    /**
     * @return string
     */
    public function getAutorizacion()
    {
        return $this->autorizacion;
    }

    /**
     * @param string $autorizacion
     */
    public function setAutorizacion($autorizacion)
    {
        $this->autorizacion = $autorizacion;
    }

    /**
     * @return string
     */
    public function getFechaAutorizacion()
    {
        return $this->fecha_autorizacion;
    }

    /**
     * @param string $fecha_autorizacion
     */
    public function setFechaAutorizacion(string $fecha_autorizacion)
    {
        $this->fecha_autorizacion = $fecha_autorizacion;
    }



    /**
     * @return text
     */
    public function getXml()
    {
        return $this->xml;
    }

    /**
     * @param text $xml
     */
    public function setXml($xml)
    {
        $this->xml = $xml;
    }

    /**
     * @return mixed
     */
    public function getAmbiente()
    {
        return $this->ambiente;
    }

    /**
     * @param mixed $ambiente
     */
    public function setAmbiente($ambiente)
    {
        $this->ambiente = $ambiente;
    }

    /**
     * @param mixed $ambiente
     */
    public function getLabelAmbiente()
    {
        if ($this->ambiente === 1) {
            return "PRUEBAS";
        } elseif ($this->ambiente === 2) {
            return "PRODUCCION";
        } else
            return "UNDEFINED";

    }

    /**
     * @return bool
     */
    public function getAnulado()
    {
        return $this->anulado;
    }

    /**
     * @param bool $anulado
     */
    public function setAnulado(bool $anulado)
    {
        $this->anulado = $anulado;
    }

    /**
     * @return bool
     */
    public function isSinrespuesta()
    {
        return $this->sinrespuesta;
    }

    /**
     * @param bool $sinrespuesta
     */
    public function setSinrespuesta(bool $sinrespuesta)
    {
        $this->sinrespuesta = $sinrespuesta;
    }

    /**
     * @return text
     */
    public function getMensajeError()
    {
        return $this->mensajeError;
    }

    /**
     * @param text $mensajeError
     */
    public function setMensajeError($mensajeError)
    {
        $this->mensajeError = $mensajeError;
    }

    /**
     * @return string
     */
    public function getTipoDocumento(): string
    {
        return $this->tipoDocumento;
    }

    /**
     * @param string $tipoDocumento
     */
    public function setTipoDocumento(string $tipoDocumento)
    {
        $this->tipoDocumento = $tipoDocumento;
    }

    /**
     * @return string
     */
    public function getAnio()
    {
        return $this->anio;
    }

    /**
     * @param string $anio
     */
    public function setAnio(string $anio)
    {
        $this->anio = $anio;
    }

    /**
     * @return string
     */
    public function getNumeroCompra()
    {
        return $this->numero_compra;
    }

    /**
     * @param string $numero_compra
     */
    public function setNumeroCompra(string $numero_compra)
    {
        $this->numero_compra = sprintf("%09d",$numero_compra);
    }

    /**
     * @return string
     */
    public function getCodigoDocSustento()
    {
        return $this->codigoDocSustento;
    }

    /**
     * @param string $codigoDocSustento
     */
    public function setCodigoDocSustento($codigoDocSustento)
    {
        $this->codigoDocSustento = $codigoDocSustento;
    }

    /**
     * @return text
     */
    public function getXmlAutorizado()
    {
        return $this->xmlAutorizado;
    }

    /**
     * @param text $xmlAutorizado
     */
    public function setXmlAutorizado($xmlAutorizado)
    {
        $this->xmlAutorizado = $xmlAutorizado;
    }

    /**
     * @return mixed
     */
    public function getDocpdf()
    {
        return $this->docpdf;
    }

    /**
     * @param mixed $docpdf
     */
    public function setDocpdf($docpdf)
    {
        $this->docpdf = $docpdf;
    }

    /**
     * @return mixed
     */
    public function getEmpresa()
    {
        return $this->empresa;
    }

    /**
     * @param mixed $empresa
     */
    public function setEmpresa($empresa)
    {
        $this->empresa = $empresa;
    }

    /**
     * @return string
     */
    public function getEstado(): string
    {
        if ($this->autorizado)
            return 'A';
        else
            return 'N';
    }
}
