<?php

namespace App\Controller\Base;

use App\Electronico\Comprobante;
use App\Entity\Model\AbstractGuiaRemision;
use App\Entity\Model\GuiaRemision;
use App\Entity\Model\Item;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use JMS\Serializer\SerializerBuilder;
use Symfony\Component\Config\Definition\Exception\Exception;
use Symfony\Component\Form\Extension\Core\DataTransformer\MoneyToLocalizedStringTransformer;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;

abstract class AbstractGuiaRemisionController extends AbstractController
{

    public $translator;

    protected function consultarAutorizacion(GuiaRemision $invoice, EntityManagerInterface $entityManager)
    {
        $respuesta = null;
        try {
            $em = $entityManager;

            $app_url = $this->getParameter('api_url');

            $error = false;

            $resp = RestApiFunciones::consultarAutoComprobante($error, $app_url, $invoice->getClaveAcceso());

            if ($error) {
                $this->addTranslatedMessage($resp, 'danger');
                $invoice->setSinrespuesta(true);
                $invoice->setMensajeError($resp);
                $em->persist($invoice);
                $em->flush();
            } elseif ($resp->autorizado) {
                $this->addTranslatedMessage('Comprobante autorizado, fecha: ' . $resp->fecha);
                $invoice->setAutorizado(true);
                $invoice->setSinrespuesta(false);
                $invoice->setMensajeError("");
                $invoice->setFechaAutorizacion($resp->fecha);
                $invoice->setXmlAutorizado($resp->comprobante);
                $invoice->setStatus(GuiaRemision::CLOSED);
                $invoice->setForcefullyClosed(true);
                $em->persist($invoice);
                $em->flush();
            } else {
                $this->addTranslatedMessage($resp, 'danger');
            }
        } catch (Exception $exception) {
            return $exception->getMessage();
        }

        return $respuesta;
    }

    public $logger;

    protected function generarXml(GuiaRemision $guia, EntityManagerInterface $entityManager)
    {
        try {
            $empresa = $guia->getEmpresa();

            $serie = $guia->getSerie();

            $numero = str_pad($guia->getNumber(), 9, "0", STR_PAD_LEFT);

            $emisor = new Comprobante(null, $empresa, $serie);

            $resp = $emisor->generaClave($guia->getIssueDate()->format('d/m/Y'), "06", $numero);

            if ($resp !== null) {
                $this->addTranslatedMessage($resp, 'danger');
                return null;
            }

            $guia->setClaveAcceso($emisor->getClaveAcceso());

            $error = false;
            $guiaremision = $emisor->generarXmlGuiaRemision($guia, $error);

            if ($error)
                $this->addTranslatedMessage($guiaremision, 'danger');

            try {
                $serializer = SerializerBuilder::create()->build();
                $xml = $serializer->serialize($guiaremision, 'xml');
                $xml = trim(preg_replace('#\s+#', ' ', $xml));
                $guia->setXml($xml);
            } catch (Exception $exception) {
                $this->addTranslatedMessage($exception->getMessage(), 'danger');
                return null;
            }

            $em = $entityManager;
            $em->persist($guia);
            $em->flush();

        } catch (Exception $exception) {
            $this->addTranslatedMessage($exception->getMessage(), 'danger');
            return null;
        }

        return $guia;
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'invoice'));
    }

    protected function bulkDelete(array $invoices, EntityManagerInterface $entityManager)
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');

        $em = $entityManager;

        foreach ($invoices as $invoice) {
            $em->remove($invoice);
        }

        $em->flush();
        $this->addTranslatedMessage('flash.bulk_deleted');

        return $this->redirectToRoute('invoice_index');
    }

    protected function bulkPdf(array $invoices)
    {
        $pages = [];
        foreach ($invoices as $invoice) {
            $pages[] = $this->getGuiaRemisionPrintPdfHtml($invoice);
        }

        $html = $this->get('siwapp_core.html_page_merger')->merge($pages, '<div class="pagebreak"> </div>');
        $pdf = $this->getPdf($html);

        return new Response($pdf, \Symfony\Component\HttpFoundation\Response::HTTP_OK, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="GuiaRemisions.pdf"'
        ]);
    }

    protected function bulkPrint(array $invoices)
    {
        $pages = [];
        foreach ($invoices as $invoice) {
            $pages[] = $this->getGuiaRemisionPrintPdfHtml($invoice, true);
        }

        $html = $this->get('siwapp_core.html_page_merger')->merge($pages, '<div class="pagebreak"> </div>');

        return new Response($html);
    }

    protected function bulkEmail(array $invoices, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;
        foreach ($invoices as $invoice) {
            $message = $this->getEmailMessage($invoice);
            $result = $this->get('mailer')->send($message);
            if ($result) {
                $invoice->setSentByEmail(true);
                $em->persist($invoice);
            }
        }

        $em->flush();
        $this->addTranslatedMessage('flash.bulk_emailed');

        return $this->redirectToRoute('invoice_index');
    }

    protected function generarPdf(GuiaRemision $invoice)
    {
        $filename = 'FAC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $xml = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url').'facturaride.php';

        $error = false;
        $mensaje = "";
        $empresa = $invoice->getEmpresa();
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            file_put_contents($filename, $docPdf);

            header("Cache-Control: public");
            header("Content-Description: File Transfer");
            header(sprintf('Content-Disposition: attachment; filename=%s', $filename));
            header("Content-Type: application/pdf");
            header('Content-Length: ' . filesize($filename));
            //header("Content-Transfer-Encoding: binary");
            header('Accept-Ranges: bytes');
            echo $docPdf;

            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }

        }
    }

    protected function delete(GuiaRemision $invoice, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;

        if ($invoice->getAutorizado() && $invoice->getAmbiente() == 2 ) {
            $this->addTranslatedMessage('Guia Remisión no puede ser eliminada, estado: AUTORIZADO, ambiente: PRODUCCION', 'warning');
            return false;
        } else {
            $em->remove($invoice);
            $em->flush();
            $this->addTranslatedMessage('Guía Remision Eliminada');

            return true;
        }
    }

    protected function enviarMail($email, GuiaRemision $invoice, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;

        $xmlAutorizado = null;

        if ($invoice->getAutorizado())
            $xmlAutorizado = $invoice->getXmlAutorizado();

        $numero = $invoice->getSerie() . '-' . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT);

        $error = false;
        $mensaje = "";

        $data = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url');

        $app_url .= 'guiaremisionride.php';

        $empresa = $invoice->getEmpresa();
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }

        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $data, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            $result = RestApiFunciones::envioMailComprobante(
                $invoice->getDestinatarios()[0]->getName(),
                $numero,
                $invoice->getClaveAcceso(),
                $invoice->getFechaAutorizacion(),
                "Factura",
                $email,
                $docPdf,
                $xmlAutorizado,
                $invoice->getEmpresa()
            );

            if ($result == null || $result === null ) {
                $this->addTranslatedMessage('MENSAJE ENVIADO A: ' . $email);
                $invoice->setSentByEmail(true);
                $em->persist($invoice);
                $em->flush();
            } else
                $this->addTranslatedMessage('ERRROR, ' . $result, 'danger');
        }
    }
}

