<?php

namespace App\Controller\Base;

use App\Electronico\Comprobante;
use App\Entity\Model\AbstractLiquidacion;
use App\Entity\Model\Empresa;
use App\Entity\Model\Liquidacion;
use App\Entity\Model\Item;
use App\Entity\Model\Provider;
use App\Entity\Model\Reembolso;
use App\Entity\Model\Tax;
use App\Service\EmFactory;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use JMS\Serializer\SerializerBuilder;
use Symfony\Component\Config\Definition\Exception\Exception;
use Symfony\Component\Form\Extension\Core\DataTransformer\MoneyToLocalizedStringTransformer;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;

abstract class AbstractLiquidacionController extends AbstractController
{

    public $translator;

    protected function cargarImpuestos(AbstractLiquidacion $liquidacion)
    {
        $basecero = 0;
        $baseiva = 0;

        foreach ($liquidacion->getItems() as $item) {

            $iva = 0;
            $p = $item->getProduct();
            $subtotal = $item->getQuantity() * $item->getUnitaryCost();

            if ($item->getTaxes() != null) {

                $tax = $item->getTaxes();
                //foreach ($postItem['taxes'] as $taxId) {
                //$tax = $taxRepo->find($taxId);
                //$tax = new Tax();
                if ($tax != null) {
                    $porcentaje = (double)$tax->getValue();
                    if ($porcentaje > 0) {
                        //continue;
                        $iva = $subtotal * ($porcentaje / 100);

                        $baseiva += $subtotal;

                        $item->setValiva($iva);
                        $item->addTax($tax);
                        $item->setSubtotal($subtotal);
                        $liquidacion->setPorIva($tax->getValue());
                    } else {
                        $basecero += $subtotal;

                        $item->setValiva(0);
                        $item->addTax($tax);
                        $item->setSubtotal($subtotal);
                        $liquidacion->setPorIva($tax->getValue());
                    }
                }
            }


            //$item->addTax($tax);
            //}
            /*} else {
                $basecero += $subtotal;

                $item->setSubtotal($subtotal);
                $item->setValIva($iva);
            }
            */
        }

        $liquidacion->setBasecero($basecero);
        $liquidacion->setBaseiva($baseiva);
    }

    protected function cargarReembolsos(AbstractLiquidacion $liquidacion, $em)
    {
        $baseImponible = 0;
        $valorIva = 0;

        foreach ($liquidacion->getReembolsos() as $reembolso) {

            //$reembolso = new Reembolso();
            $providers = $em->getRepository(\App\Entity\Model\Provider::class)->findBy([
                'name' => $reembolso->getProvreembolsoName(),
                'identification' => $reembolso->getProvreembolsoIdentification(),
            ]);
            $provider = reset($providers);
            if ($provider) {
                $reembolso->setProvider($provider);
            }

            $reembolso->setLiquidacion($liquidacion);
            $subtotal = $reembolso->getBaseAmount();

            if ($reembolso->getTaxes() != null) {

                $tax = $reembolso->getTaxes();
                //foreach ($postItem['taxes'] as $taxId) {
                //$tax = $taxRepo->find($taxId);
                //$tax = new Tax();
                if ($tax != null) {
                    $porcentaje = (double)$tax->getValue();
                    if ($porcentaje > 0) {
                        //continue;
                        $iva = $subtotal * ($porcentaje / 100);

                        $valorIva += $iva;

                        $reembolso->setValiva($iva);
                        $reembolso->addTax($tax);
                        $reembolso->setImponible($subtotal);
                        $reembolso->setPorIva($tax->getValue());
                        $reembolso->setBaseIva($subtotal);
                    } else {

                        $reembolso->setValiva(0);
                        $reembolso->addTax($tax);
                        $reembolso->setImponible($subtotal);
                        $reembolso->setPorIva($tax->getValue());
                        $reembolso->setBaseCero($subtotal);
                    }
                }
            }

            $baseImponible += $subtotal;
        }

        $liquidacion->setBaseReemboAmount($baseImponible);
        $liquidacion->setTaxReemboAmount($valorIva);
        $liquidacion->setGrossReemboAmount($baseImponible+$valorIva);
    }


    protected function consultarAutorizacion(Liquidacion $liquidacion, EntityManagerInterface $entityManager)
    {
        $respuesta = null;
        try {
            $em = $entityManager;

            $app_url = $this->getParameter('api_url');

            $error = false;

            $resp = RestApiFunciones::consultarAutoComprobante($error, $app_url, $liquidacion->getClaveAcceso());

            if ($error) {
                $this->addTranslatedMessage($resp, 'danger');
                $liquidacion->setSinrespuesta(true);
                $liquidacion->setMensajeError($resp);
                $em->persist($liquidacion);
                $em->flush();
            } elseif ($resp->autorizado) {
                $this->addTranslatedMessage('Comprobante autorizado, fecha: ' . $resp->fecha);
                $liquidacion->setAutorizado(true);
                $liquidacion->setSinrespuesta(false);
                $liquidacion->setMensajeError("");
                $liquidacion->setFechaAutorizacion($resp->fecha);
                $liquidacion->setXmlAutorizado($resp->comprobante);
                $liquidacion->setStatus(Liquidacion::CLOSED);
                $liquidacion->setForcefullyClosed(true);
                $em->persist($liquidacion);
                $em->flush();
            } else {
                $this->addTranslatedMessage($resp, 'danger');
            }
        } catch (Exception $exception) {
            return $exception->getMessage();
        }

        return $respuesta;
    }

    public $logger;

    protected function generarXml(Liquidacion $liquidacion, EntityManagerInterface $entityManager)
    {
        try {
            $empresa = $liquidacion->getEmpresa();

            $serie = $liquidacion->getSerie();

            $numero = str_pad($liquidacion->getNumber(), 9, "0", STR_PAD_LEFT);

            $emisor = new Comprobante(null, $empresa, $serie);

            $resp = $emisor->generaClave($liquidacion->getIssueDate()->format('d/m/Y'), "03", $numero);

            if ($resp !== null) {
                $this->addTranslatedMessage($resp, 'danger');
                return null;
            }

            $liquidacion->setClaveAcceso($emisor->getClaveAcceso());

            $error = false;
            $liquidacionCompra = $emisor->generarXmlLiquidacion($liquidacion, $error);

            if ($error)
                $this->addTranslatedMessage($liquidacion, 'danger');

            try {
                $serializer = SerializerBuilder::create()->build();
                $xml = $serializer->serialize($liquidacionCompra, 'xml');
                $xml = trim(preg_replace('#\s+#', ' ', $xml));
                $liquidacion->setXml($xml);
            } catch (Exception $exception) {
                $this->addTranslatedMessage($exception->getMessage(), 'danger');
                return null;
            }

            $em = $entityManager;
            $em->persist($liquidacion);
            $em->flush();

        } catch (Exception $exception) {
            $this->addTranslatedMessage($exception->getMessage(), 'danger');
            return null;
        }

        return $liquidacion;
    }

    protected function getLiquidacionTotalsFromPost(array $post, AbstractLiquidacion $liquidacion, string $locale, EntityManagerInterface $entityManager): array
    {
        $user = $this->getUser();

        $em = $entityManager;

        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $taxRepo = $em->getRepository(Tax::class);
        //$currency = $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR');
        $currency = $empresa == null ? 'USD' : $empresa->getCurrency();
        $formatter = new \NumberFormatter($locale, \NumberFormatter::CURRENCY);
        $transformer = new MoneyToLocalizedStringTransformer(2, true);

        $totals = [];
        foreach ($post['items'] as $index => $postItem) {
            $item = new Item($taxRepo->findTaxDefault($empresa->getId()));
            $item->setUnitaryCost($transformer->reverseTransform($postItem['unitary_cost']));
            $item->setQuantity($postItem['quantity']);
            $item->setDiscount($postItem['discount_percent']);
            if (isset($postItem['taxes'])) {
                //foreach($postItem['taxes'] as $taxId) {
                $tax = $taxRepo->find($postItem['taxes']);
                if (!$tax instanceof \App\Entity\Model\Tax) {
                    continue;
                }

                $item->setTaxes($tax);
                //}

                //$item->setTaxes($postItem['taxes']);
            }

            $totals['items'][$index] = [
                'gross_amount' => $formatter->formatCurrency($item->getGrossAmount(), $currency),
            ];
            $liquidacion->addItem($item);
        }

        $liquidacion->checkAmounts();

        $pagoTotal = 0;
        foreach ($post['pagos'] as $index => $pago) {
            $pagoTotal += $transformer->reverseTransform($pago['valor']);
        }

        return $totals + [
            'liquidacion_base_amount' => $formatter->formatCurrency($liquidacion->getBaseAmount(), $currency),
            'liquidacion_tax_amount' => $formatter->formatCurrency($liquidacion->getTaxAmount(), $currency),
            'liquidacion_gross_amount' => $formatter->formatCurrency($liquidacion->getGrossAmount(), $currency),
            'pago_gross_amount' => number_format($liquidacion->getGrossAmount(), 2),
        ];
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'liquidacion'));
    }

    protected function bulkDelete(array $liquidacions, EntityManagerInterface $entityManager)
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');

        $em = $entityManager;

        foreach ($liquidacions as $liquidacion) {
            $em->remove($liquidacion);
        }

        $em->flush();
        $this->addTranslatedMessage('flash.bulk_deleted');

        return $this->redirectToRoute('liquidacion_index');
    }

    protected function bulkPdf(array $liquidacions)
    {
        $pages = [];
        foreach ($liquidacions as $liquidacion) {
            $pages[] = $this->getLiquidacionPrintPdfHtml($liquidacion);
        }

        $html = $this->get('siwapp_core.html_page_merger')->merge($pages, '<div class="pagebreak"> </div>');
        $pdf = $this->getPdf($html);

        return new Response($pdf, \Symfony\Component\HttpFoundation\Response::HTTP_OK, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="Liquidacions.pdf"'
        ]);
    }

    protected function bulkPrint(array $liquidacions)
    {
        $pages = [];
        foreach ($liquidacions as $liquidacion) {
            $pages[] = $this->getLiquidacionPrintPdfHtml($liquidacion, true);
        }

        $html = $this->get('siwapp_core.html_page_merger')->merge($pages, '<div class="pagebreak"> </div>');

        return new Response($html);
    }

    protected function bulkEmail(array $liquidacions, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;
        foreach ($liquidacions as $liquidacion) {
            $message = $this->getEmailMessage($liquidacion);
            $result = $this->get('mailer')->send($message);
            if ($result) {
                $liquidacion->setSentByEmail(true);
                $em->persist($liquidacion);
            }
        }

        $em->flush();
        $this->addTranslatedMessage('flash.bulk_emailed');

        return $this->redirectToRoute('liquidacion_index');
    }

    protected function generarPdf(Liquidacion $liquidacion)
    {
        $filename = 'LC_' . $liquidacion->getSerie() . "-" . str_pad($liquidacion->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $xml = $liquidacion->getAutorizado() ? $liquidacion->getXmlAutorizado() : $liquidacion->getXml();

        $app_url = $this->getParameter('api_url').'liquidacionride.php';

        $error = false;
        $mensaje = "";
        $empresa = $liquidacion->getEmpresa();
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $liquidacion->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            file_put_contents($filename, $docPdf);

            header("Cache-Control: public");
            header("Content-Description: File Transfer");
            header(sprintf('Content-Disposition: attachment; filename=%s', $filename));
            header("Content-Type: application/pdf");
            header('Content-Length: ' . filesize($filename));
            //header("Content-Transfer-Encoding: binary");
            header('Accept-Ranges: bytes');
            echo $docPdf;

            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }

        }
    }

    protected function delete(Liquidacion $liquidacion, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;

        if ($liquidacion->getAutorizado() && $liquidacion->getAmbiente() == 2) {
            $this->addTranslatedMessage('Factura no puede ser eliminada, estado: AUTORIZADO, ambiente: PRODUCCION', 'warning');
            return false;
        } else {
            $em->remove($liquidacion);
            $em->flush();
            $this->addTranslatedMessage('flash.deleted');

            return true;
        }
    }

    protected function enviarMail($email, Liquidacion $liquidacion, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;

        $xmlAutorizado = null;

        if ($liquidacion->getAutorizado())
            $xmlAutorizado = $liquidacion->getXmlAutorizado();

        $numero = $liquidacion->getSerie() . '-' . str_pad($liquidacion->getNumber(), 9, '0', STR_PAD_LEFT);

        $error = false;
        $mensaje = "";

        //$filename = 'FAC_' . $liquidacion->getSerie() . "-" . str_pad($liquidacion->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $data = $liquidacion->getAutorizado() ? $liquidacion->getXmlAutorizado() : $liquidacion->getXml();

        $app_url = $this->getParameter('api_url');

        $app_url .= 'liquidacionride.php';
        $empresa = $liquidacion->getEmpresa();
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $liquidacion->getClaveAcceso(), $data, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            $result = RestApiFunciones::envioMailComprobante(
                $liquidacion->getProviderName(),
                $numero,
                $liquidacion->getClaveAcceso(),
                $liquidacion->getFechaAutorizacion(),
                "Liquidacion",
                $email,
                $docPdf,
                $xmlAutorizado,
                $liquidacion->getEmpresa()
            );

            if ($result == null || $result === null ) {
                $this->addTranslatedMessage('MENSAJE ENVIADO A: ' . $email);
                $liquidacion->setSentByEmail(true);
                $em->persist($liquidacion);
                $em->flush();
            } else
                $this->addTranslatedMessage('ERRROR, ' . $result, 'danger');
        }
    }

    protected function getLiquidacionReembolsosTotalsFromPost(array $post, AbstractLiquidacion $liquidacion, string $locale, EntityManagerInterface $entityManager): array
    {
        $user = $this->getUser();

        $em = $entityManager;

        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $taxRepo = $em->getRepository(\App\Entity\Model\Tax::class);
        //$currency = $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR');
        $currency = $empresa == null ? 'USD' : $empresa->getCurrency();
        $formatter = new \NumberFormatter($locale, \NumberFormatter::CURRENCY);
        $transformer = new MoneyToLocalizedStringTransformer(2, true);

        $totals = [];
        foreach ($post['reembolsos'] as $index => $postItem) {
            $item = new Reembolso();
            $item->setImponible($transformer->reverseTransform($postItem['imponible']));
            //$item->setQuantity($postItem['quantity']);
            if (isset($postItem['taxes'])) {
                //foreach($postItem['taxes'] as $taxId) {
                $tax = $taxRepo->find($postItem['taxes']);
                if (!$tax instanceof \App\Entity\Model\Tax) {
                    continue;
                }

                $item->setTaxes($tax);
                //}

                //$item->setTaxes($postItem['taxes']);
            }

            $totals['items'][$index] = [
                'gross_reembo_amount' => $formatter->formatCurrency($item->getImponible(), $currency),
            ];
            $liquidacion->addReembolso($item);
        }

        $liquidacion->checkReemboAmounts();

        return $totals + [
            'base_reembo_amount' => $formatter->formatCurrency($liquidacion->getBaseReemboAmount(), $currency),
            'tax_reembo_amount' => $formatter->formatCurrency($liquidacion->getTaxReemboAmount(), $currency),
            'gross_reembo_amount' => $formatter->formatCurrency($liquidacion->getGrossReemboAmount(), $currency),
        ];
    }

}

