<?php

namespace App\Controller;

use App\Controller\Base\AbstractInvoiceController;
use App\Entity\Model\Customer;
use App\Entity\Model\Empresa;
use App\Entity\Model\Invoice;
use App\Entity\Model\Item;
use App\Entity\Model\ItemInvoiceReembolso;
use App\Entity\Model\ItemPago;
use App\Entity\Model\Provider;
use App\Entity\Model\Tax;
use App\Form\CustomerType;
use App\Form\InvoiceReembolsoType;
use App\Form\ItemInvoiceReembolsoType;
use App\Form\ListGenericType;
use App\Form\ProviderType;
use App\Form\SearchGenericType;
use App\Service\EmFactory;
use App\Util\Funciones;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use Knp\Component\Pager\PaginatorInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Form\Extension\Core\Type\HiddenType;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Contracts\Translation\TranslatorInterface;


/**
 * @Route("/invoreembolso")
 */
class InvoiceReembolsoController extends AbstractInvoiceController
{

    public $translator;

    /**
     * @Route("", name="invoreembolso_index")
     *
     */
    public function indexAction(EmFactory $emFactory, Request $request, PaginatorInterface $paginator, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $emisor = $empresaRepo->findOneByUser($user->getRuc());

        $repo = $em->getRepository(Invoice::class);
        $repo->setPaginator($paginator);
        // @todo Unhardcode this.
        $limit = 50;

        $form = $this->createForm(SearchGenericType::class, null, [
            'action' => $this->generateUrl('invoreembolso_index'),
            'method' => 'GET',
        ]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $pagination = $repo->paginatedSearchReembolso($form->getData(), $limit, $request->query->getInt('page', 1), $emisor->getId());
        } else {
            $pagination = $repo->paginatedSearchReembolso([], $limit, $request->query->getInt('page', 1), $emisor->getId());
        }

        $invoices = [];
        foreach ($pagination->getItems() as $item) {
            $invoices[] = $item[0];
        }

        $listForm = $this->createForm(ListGenericType::class, $invoices, [
            'action' => $this->generateUrl('invoreembolso_index'),
        ]);

        $listForm->handleRequest($request);

        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            if (empty($data['invoices'])) {
                $this->addTranslatedMessage('flash.nothing_selected', 'warning');
            } elseif ($request->request->has('delete')) {
                return $this->bulkDelete($data['invoices']);
            } elseif ($request->request->has('pdf')) {
                return $this->bulkPdf($data['invoices']);
            } elseif ($request->request->has('print')) {
                return $this->bulkPrint($data['invoices']);
            } elseif ($request->request->has('email')) {
                return $this->bulkEmail($data['invoices']);
            }
        }

        return $this->render('InvoiceReembolso\index.html.twig',
            [
                'invoices' => $pagination,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
                'search_form' => $form->createView(),
                'list_form' => $listForm->createView(),
            ]);
    }


    /**
     * @Route("/show/{slug}liq{id}", name="invoreembolso_show")
     *
     */
    public function showAction($id, EmFactory $emFactory, $slug, TranslatorInterface $translator, Request $request, LoggerInterface $logger): \Symfony\Component\HttpFoundation\Response
    {
        $this->logger = $logger;
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $emisor = $empresaRepo->findOneByUser($user->getRuc());

        $entity = $em->getRepository(Invoice::class)->findBySlug($slug, $id);
        if (!$entity) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        $defaultData = ['slug' => $slug, 'id'=>$id, 'customerEmail' => $entity->getCustomerEmail()];

        $form = $this->createFormBuilder($defaultData)
            ->add('slug', HiddenType::class)
            ->add('id', HiddenType::class)
            ->add('customerEmail')
            ->setAction($this->generateUrl('invoreembolso_show', ['id' => $id, 'slug'=>$slug]))
            ->getForm();

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            // data is an array with "name", "email", and "message" keys
            $data = $form->getData();

            if ($request->request->has('Form-pdf')) {
                $this->generarPdf($entity);
            } elseif ($request->request->has('Form-email')) {
                if (isset($data['customerEmail'])) {
                    $email = $data['customerEmail'];
                    $this->enviarMail($email, $entity, $em);
                } else
                    $this->addTranslatedMessage('Email del cliente nulo o en blanco', 'warning');
            } elseif ($request->request->has('Form-edit')) {
                if ($entity->getAutorizado() === false)
                    return $this->redirectToRoute('invoreembolso_edit', ['id' => $entity->getId(), 'slug' => $slug]);
            } elseif ($request->request->has('Form-delete')) {
                $error = $this->delete($entity, $em);
                if ($error)
                    return $this->redirectToRoute('invoreembolso_index');
            } elseif ($request->request->has('Form-anular')) {
                $entity->setAnulado(true);
                $em->persist($entity);
                $em->flush();
            } elseif ($request->request->has('Form-enviar')) {
                $this->enviarSriOnline($entity, $em);
            } elseif ($request->request->has('Form-auto')) {
                if ($entity->getAutorizado() === false) {
                    $resp = $this->consultarAutorizacion($entity, $em);
                    if ($resp != null)
                        $this->addTranslatedMessage($resp, 'danger');
                }

            }
        }

        if ($entity->getMensajeError()) {
            $this->addTranslatedMessage($entity->getMensajeError(), 'danger');
        }


        return $this->render('InvoiceReembolso\show.html.twig',
            [
                'entity' => $entity,
                'form' => $form->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor == null ? 'USD' : $emisor->getCurrency(),
                'decpunit' => $emisor->getDecPunit(),
            ]);
    }

    /**
     * @Route("/new", name="invoreembolso_add")
     *
     */
    public function newAction(EmFactory $emFactory, Request $request, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $msg='';
        $puedefacturar = Funciones::getValidaPuedeFacturar($empresa, $em, $msg);
        if ($empresa->getPuedefacturar() === false && $empresa->getTipoAmbiente() === "2") {
            $this->addTranslatedMessage($msg, 'danger');
            return $this->redirect($this->generateUrl('invoreembolso_index'));
        }

        $empresa = Funciones::getValidaEmitidos($empresa, $em);
        if ($empresa->getPuedefacturar() === false && $empresa->getTipoAmbiente() === "2") {
            $this->addTranslatedMessage('Número de comprobantes emitidos excede del plan contratado ', 'danger');
            return $this->redirect($this->generateUrl('invoreembolso_index'));
        }

        $invoice = new Invoice();
        $invoice->setEsReembolso(true);

        $newItem = new Item();

        $tax = $em->getRepository(Tax::class)->findTaxDefault($empresa->getId());
        $newItem->setTaxes($tax);
        $invoice->addItem($newItem);

        $itemPago = new ItemPago();
        $itemPago->setPlazo(0);

        $invoice->addPago($itemPago);

        $reembolso = new ItemInvoiceReembolso();
        $invoice->addReembolso($reembolso);

        $form = $this->createForm(InvoiceReembolsoType::class, $invoice, [
            'action' => $this->generateUrl('invoreembolso_add'),
        ]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            if ($request->request->has('save_draft')) {
                $invoice->setStatus(Invoice::DRAFT);
            } else {
                // Any save action transforms this to opened.
                $invoice->setStatus(Invoice::OPENED);
            }

            $serie = $invoice->getSeries()->getValue();
            $invoice->setSerie($serie);
            $this->cargarImpuestos($invoice);
            $this->cargarReembolsos($invoice, $em);
            $invoice->setEmpresa($empresa);
            $invoice->setUsuario($user->getId());
            $invoice->setAmbiente($empresa->getTipoAmbiente());

            $em->persist($invoice);
            $em->flush();
            //$this->addTranslatedMessage('flash.added');

            $invoice_id = $invoice->getId();

            $invoice = $this->generarXml($invoice, $em);

            if ($empresa->getEnvioAutomatico()) {

               /* if ($liquidacion !== null) {
                    $app_url = $this->getParameter('cola_url');

                    $error = false;

                    $resp = RestApiFunciones::enviarComprobanteCola($error,
                        $liquidacion->getProviderName(),
                        $liquidacion->getProviderEmail(),
                        $user->getId(),
                        $liquidacion->getXml(),
                        $liquidacion->getClaveAcceso(),
                        $app_url,
                        'mifactureroecinvoice'
                    );

                    if ($error) {
                        $this->addTranslatedMessage($resp, 'danger');
                        $liquidacion->setMensajeError($resp);
                    }

                    $em->persist($liquidacion);
                    $em->flush();

                }

                return $this->redirectToRoute('liqreembolso_index');*/

            }

            return $this->redirectToRoute('invoreembolso_show', ['id' => $invoice_id, 'slug' => $invoice->getSlug()]);

        }

        $provider = new Provider();

        $formprovider = $this->createForm(ProviderType::class, $provider, [
            'action' => $this->generateUrl('rest_provider_add'),
        ]);
        $formprovider->handleRequest($request);

        $formprovreembolso = $this->createForm(ProviderType::class, $provider, [
            'action' => $this->generateUrl('rest_provider_add'),
        ]);
        $formprovreembolso->handleRequest($request);

        $reembolso = new ItemInvoiceReembolso();
        $formreembolso = $this->createForm(itemInvoiceReembolsoType::class, $reembolso, [
            'action' => $this->generateUrl('rest_provider_add'),
        ]);
        $formreembolso->handleRequest($request);

        $customer = new Customer();

        $formcustomer = $this->createForm(CustomerType::class, $customer, [
            'action' => $this->generateUrl('rest_customer_add'),
        ]);
        $formcustomer->handleRequest($request);

        return $this->render('InvoiceReembolso\edit.html.twig',
            [
                'form' => $form->createView(),
                'formprovider' => $formprovider->createView(),
                'formreembolso' => $formreembolso->createView(),
                'formprovreembolso' => $formprovreembolso->createView(),
                'provider' => $provider,
                'entity' => $invoice,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),
                'formcustomer' => $formcustomer->createView(),

            ]);
    }


    /**
     * @Route("/edit/{slug}liq{id}", name="invoreembolso_edit")
     *
     */
    public function editAction($id, EmFactory $emFactory, Request $request, $slug, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoice = $em->getRepository(Invoice::class)->findBySlug($slug, $id);
        if (!$invoice && $invoice->getEmpresa() != $empresa) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        if($invoice->getAutorizado()){
            return $this->redirectToRoute('invoreembolso_show', ['id' => $id, 'slug' => $slug]);
        }

        $form = $this->createForm(InvoiceReembolsoType::class, $invoice, [
            'action' => $this->generateUrl('invoreembolso_edit', ['id' => $id, 'slug' => $slug]),
        ]);
        $form->handleRequest($request);

        $invoice->setEmpresa($empresa);

        if ($form->isSubmitted() && $form->isValid()) {
            $redirectRoute = 'liqreembolso_show';

            $serie = $invoice->getSeries()->getValue();
            $invoice->setSerie($serie);
            $this->cargarImpuestos($invoice);
            $this->cargarReembolsos($invoice, $em);
            $invoice->setEmpresa($empresa);
            $invoice->setUsuario($user->getId());
            $invoice->setAmbiente($empresa->getTipoAmbiente());

            $em->persist($invoice);
            $em->flush();
            //$this->addTranslatedMessage('flash.added');

            $invoice_id = $invoice->getId();

            $invoice = $this->generarXml($invoice, $em);

            if ($empresa->getEnvioAutomatico()) {

               /* if ($liquidacion !== null) {
                    $app_url = $this->getParameter('cola_url');

                    $error = false;

                    $resp = RestApiFunciones::enviarComprobanteCola($error,
                        $liquidacion->getProviderName(),
                        $liquidacion->getProviderEmail(),
                        $user->getId(),
                        $liquidacion->getXml(),
                        $liquidacion->getClaveAcceso(),
                        $app_url,
                        'mifactureroecinvoice'
                    );

                    if ($error) {
                        $this->addTranslatedMessage($resp, 'danger');
                        $liquidacion->setMensajeError($resp);
                    }

                    $em->persist($liquidacion);
                    $em->flush();

                }

                return $this->redirectToRoute('liqreembolso_index');*/

            }

            return $this->redirectToRoute('invoreembolso_show', ['id' => $invoice_id, 'slug' => $invoice->getSlug()]);
        }

        $provider = new Provider();

        $formprovider = $this->createForm(ProviderType::class, $provider, [
            'action' => $this->generateUrl('rest_provider_add'),
        ]);
        $formprovider->handleRequest($request);

        $formprovreembolso = $this->createForm(ProviderType::class, $provider, [
            'action' => $this->generateUrl('rest_provider_add'),
        ]);
        $formprovreembolso->handleRequest($request);

        $reembolso = new ItemInvoiceReembolso();
        $formreembolso = $this->createForm(ItemInvoiceReembolsoType::class, $reembolso, [
            'action' => $this->generateUrl('rest_provider_add'),
        ]);
        $formreembolso->handleRequest($request);
        return $this->render('InvoiceReembolso\edit.html.twig',
            [
                'entity' => $invoice,
                'form' => $form->createView(),
                'formprovider' => $formprovider->createView(),
                'formreembolso' => $formreembolso->createView(),
                'formprovreembolso' => $formprovreembolso->createView(),
                'provider' => $provider,
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $empresa == null ? 'USD' : $empresa->getCurrency(),

            ]);
    }

    public function enviarSriOnline(Invoice $invoice, EntityManagerInterface $em)
    {
        $invoice = $this->generarXml($invoice, $em);

        if($invoice !== null) {
            $app_url = $this->getParameter('api_url');

            $error = false;
            $resp = RestApiFunciones::enviarComprobante($error, $invoice->getXml(), $invoice->getClaveAcceso(), $app_url);

            if ($error) {
                $this->addTranslatedMessage($resp->message, 'danger');
            } elseif ($resp->estado === 'recibido') {
                $resp = $this->consultarAutorizacion($invoice, $em);
                if ($resp !== null)
                    $this->addTranslatedMessage($resp, 'danger');
            } else {
                $invoice->setMensajeError($resp->message);
                $this->addTranslatedMessage($resp->message, 'danger');
                $em->persist($invoice);
                $em->flush();
            }

        }
    }

    /**
     * @Route("/payments/{slug}liq{id}", name="liqreembolso_payments")
     *
     */
    public function paymentsAction(Request $request, $slug, EmFactory $emFactory, TranslatorInterface $translator): \Symfony\Component\HttpFoundation\Response
    {
        $emisor = null;
        $this->translator = $translator;

        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        // Return all payments
        $liquidacion = $em->getRepository(Liquidacion::class)->findBySlug($slug);
        if (!$liquidacion) {
            throw $this->createNotFoundException('Unable to find Liquidacion entity.');
        }

        $payment = new Payment;
        $addForm = $this->createForm(PaymentType::class, $payment, [
            'action' => $this->generateUrl('liqreembolso_payments', ['id' => $liquidacion->getId(), 'slug' => $slug]),
        ]);
        $addForm->handleRequest($request);
        if ($addForm->isSubmitted() && $addForm->isValid()) {
            $liquidacion->addPayment($payment);
            $em->persist($liquidacion);
            $em->flush();
            $this->addTranslatedMessage('payment.flash.added');

            // Rebuild the query, since we have new objects now.
            return $this->redirectToRoute('liqreembolso_index');
        }

        $listForm = $this->createForm('App\Form\ListLiquidacionPaymentType', $liquidacion->getPayments()->getValues(), [
            'action' => $this->generateUrl('liqreembolso_payments', ['id' => $liquidacion->getId(), 'slug' => $slug]),
        ]);
        $listForm->handleRequest($request);

        if ($listForm->isSubmitted() && $listForm->isValid()) {
            $data = $listForm->getData();
            foreach ($data['payments'] as $payment) {
                $liquidacion->removePayment($payment);
                $em->persist($liquidacion);
                $em->flush();
            }

            $this->addTranslatedMessage('payment.flash.bulk_deleted');

            // Rebuild the query, since some objects are now missing.
            return $this->redirectToRoute('liqreembolso_index');
        }

        return $this->render('Payment\list.html.twig',
            [
                'liquidacionId' => $liquidacionId,
                'add_form' => $addForm->createView(),
                'list_form' => $listForm->createView(),
                //'currency' => $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR'),
                'currency' => $emisor === null ? 'USD' : $emisor->getCurrency(),
            ]);
    }

    /**
     * @Route("/form-totals", name="invoreembolso_form_totals")
     */
    public function getInvoiceReembolsoFormTotals(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();

        $post = $request->request->get('invoice_reembolso');
        if (!$post) {
            throw new NotFoundHttpException;
        }

        $response = $this->getInvoiceReembolsoTotalsFromPost($post, new Invoice, $request->getLocale(), $em);

        return new JsonResponse($response);
    }

    /**
     * @Route("/items/form-totals", name="itemsinvoicereembolso_form_totals")
     */
    public function getInvoiceItemReembolsoFormTotals(EmFactory $emFactory, Request $request): \Symfony\Component\HttpFoundation\Response
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();

        $post = $request->request->get('invoice_reembolso');
        if (!$post) {
            throw new NotFoundHttpException;
        }

        $response = $this->getItemsInvoiceReembolsosTotalsFromPost($post, new Invoice, $request->getLocale(), $em);

        return new JsonResponse($response);
    }

    /**
     * @Route("/pdfpreview/{slug}inv{id}", name="invoicereembo_show_pdf_preview")
     *
     */
    public function showOnlinePdfAction($id,  $slug, EmFactory $emFactory, TranslatorInterface $translator, Request $request, LoggerInterface $logger): \Symfony\Component\HttpFoundation\Response
    {
        $this->logger = $logger;
        $this->translator = $translator;
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');
        $user = $this->getUser();

        $em = $emFactory->getEm();
        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $invoice = $em->getRepository(Invoice::class)->findBySlug($slug, $id);
        if (!$invoice) {
            throw $this->createNotFoundException('Unable to find Invoice entity.');
        }

        $filename = 'FAC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $xml = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url').'facturaride.php';

        $error = false;
        $mensaje = "";
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            file_put_contents($filename, $docPdf);

            $pdf = base64_encode(file_get_contents($filename));

            $response = new Response($pdf);
            $response->headers->set('Content-Type', 'application/octet-stream');
            $response->headers->set('Content-Description', 'File Transfer');
            $response->headers->set('Content-Disposition', 'attachment; filename="'.$filename.'"');
            // $response->headers->set('Expires', '0');
            // $response->headers->set('Content-Transfer-Encoding', 'binary');
            $response->headers->set('Content-length', strlen($pdf));
            $response->headers->set('Cache-Control', 'no-cache private');
            // $response->headers->set('Pragma', 'public');
            // Send headers before outputting anything
            $response->sendHeaders();



            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }

            return $response;
        }
    }
}
