<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Gedmo\Timestampable\Traits\TimestampableEntity;
use Gedmo\Timestampable\Timestampable;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * App\Entity\Model\AbstractRetencion
 *
 * @ORM\MappedSuperclass
 * @ORM\HasLifecycleCallbacks
 */
abstract class AbstractRetencion implements Timestampable
{
    /**
     * @var \Doctrine\Common\Collections\ArrayCollection|\App\Entity\Model\ItemRetencion[]|mixed|mixed[]
     */
    public $items;

    public $empresa;

    use TimestampableEntity;

    /**
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private  $id = null;

    /**
     * @ORM\Column(name="slug", type="string", length=32, nullable=true, unique=true)
     */
    private  $slug = null;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Provider", inversedBy="retenciones")
     * @ORM\JoinColumn(name="provider_id", referencedColumnName="id")
     */
    protected $provider;

    /**
     *
     * @ORM\Column(name="provider_name", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el nombre del proveedor")
     */
    private  $provider_name = null;


    /**
     *
     * @ORM\Column(name="provider_identification", type="string", length=128, nullable=true)
     * @Assert\NotBlank(message = "Ingrese el número de identificación del proveedor")
     */
    private  $provider_identification = null;

    /**
     *
     * @ORM\Column(name="provider_email", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese el correo electrónico del proveedor")
     * @Assert\Email()
     */
    private  $provider_email = null;

    /**
     * @ORM\Column(name="provider_address", type="string", length=255, nullable=true)
     */
    private  $provider_address = null;

    /**
     * @ORM\Column(name="notes", type="string", length=255, nullable=true)
     */
    private  $notes = null;

    /**
     * @var decimal $base_amount
     *
     * @ORM\Column(name="base_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $base_amount = null;

    /**
     * @var decimal $gross_amount
     *
     * @ORM\Column(name="gross_amount", type="decimal", scale=2, precision=15, nullable=true)
     */
    private $gross_amount = null;


    /**
     * @var smallint $status
     *
     * @ORM\Column(name="status", type="smallint", nullable=true)
     * @Assert\Length(min=0, max=3)
     */
    protected $status = 0;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Series")
     * @ORM\JoinColumn(name="series_id", referencedColumnName="id")
     * @Assert\NotBlank()
     *
     * unidirectional many-to-one
     */
    private $series;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private  $usuario = null;

    public function __construct()
    {
        $this->items = new ArrayCollection();
    }

    /**
     * Get id
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set provider parameters from provider entity.
     *
     * @param App\Entity\Model\Provider $provider
     */
    public function setFromProvider(Provider $provider)
    {
        if (empty($this->provider_name) && $provider->getName()) {
            $this->provider_name = $provider->getName();
        }

        if (empty($this->provider_email) && $provider->getEmail()) {
            $this->provider_email = $provider->getEmail();
        }

        if (empty($this->provider_identification) && $provider->getIdentification()) {
            $this->provider_identification = $provider->getIdentification();
        }

        if (empty($this->provider_address) && $provider->getAddress()) {
            $this->provider_address = $provider->getAddress();
        }

    }

    /**
     * Set provider.
     *
     * @param App\Entity\Model\Provider $provider
     */
    public function setProvider(Provider $provider)
    {
        $this->provider = $provider;
    }

    /**
     * Set provider_name
     *
     * @param string $providerName
     */
    public function setProviderName($providerName)
    {
        $this->provider_name = $providerName;
    }


    /**
     * Get provider_name
     *
     * @return string
     */
    public function getProviderName()
    {
        return $this->provider_name;
    }


    /**
     * Set provider_identification
     *
     * @param string $providerIdentification
     */
    public function setProviderIdentification($providerIdentification)
    {
        $this->provider_identification = $providerIdentification;
    }

    /**
     * Get provider_identification
     *
     * @return string
     */
    public function getProviderIdentification()
    {
        return $this->provider_identification;
    }

    /**
     * Set provider_email
     *
     * @param string $providerEmail
     */
    public function setProviderEmail($providerEmail)
    {
        $this->provider_email = $providerEmail;
    }

    /**
     * Get provider_email
     *
     * @return string
     */
    public function getProviderEmail()
    {
        return $this->provider_email;
    }

    /**
     * Set provider_address
     *
     * @param string $providerAddress
     */
    public function setProviderAddress($providerAddress)
    {
        $this->provider_address = $providerAddress;
    }

    /**
     * Get provider_address
     *
     * @return string
     */
    public function getProviderAddress()
    {
        return $this->provider_address;
    }

    /**
     * Set notes
     *
     * @param string $notes
     */
    public function setNotes($notes)
    {
        $this->notes = strtoupper($notes);
    }

    /**
     * Get notes
     *
     * @return string
     */
    public function getNotes()
    {
        return $this->notes;
    }

    /**
     * Set base_amount
     *
     * @param decimal $baseAmount
     */
    public function setBaseAmount($baseAmount)
    {
        $this->base_amount = $baseAmount;
    }

    /**
     * Get base_amount
     *
     * @return decimal
     */
    public function getBaseAmount()
    {
        return $this->base_amount;
    }

    /**
     * Set gross_amount
     *
     * @param decimal $grossAmount
     */
    public function setGrossAmount($grossAmount)
    {
        $this->gross_amount = $grossAmount;
    }


    /**
     * Get gross_amount
     *
     * @return decimal
     */
    public function getGrossAmount()
    {
        return $this->gross_amount;
    }

    /**
     * Get gross_amount
     *
     * @return decimal
     */
    public function getPorcentajeAmount()
    {
        return 0.00;
    }

    /**
     * Set status
     *
     * @param integer $status
     */
    public function setStatus($status)
    {
        $this->status = $status;
    }

    /**
     * Get status
     *
     * @return integer
     */
    public function getStatus()
    {
        return $this->status;
    }

    /**
     * Add items
     *
     * @param App\Entity\Model\ItemRetencion $item
     */
    public function addItem(ItemRetencion $item)
    {
        $this->items[] = $item;
    }

    /**
     * Get items
     *
     * @return Doctrine\Common\Collections\Collection
     */
    public function getItems()
    {
        return $this->items;
    }

    /**
     * Set serie
     *
     * @param App\Entity\Model\Series $series
     */
    public function setSeries(Series $series)
    {
        $this->series = $series;
    }

    /**
     * Get serie
     *
     * @return App\Entity\Model\Series
     */
    public function getSeries()
    {
        return $this->series;
    }

    /**
     * @return Provider
     */
    public function getProvider()
    {
        return $this->provider;
    }

    /**
     * @return mixed
     */
    public function getUsuario()
    {
        return $this->usuario;
    }

    /**
     * @param mixed $usuario
     */
    public function setUsuario($usuario)
    {
        $this->usuario = $usuario;
    }

    /**
     * @return string
     */
    public function getSlug(): string
    {
        return $this->slug;
    }



    /** ########### CUSTOM METHODS ################## */


    public function __isset($name)
    {
        return array_key_exists($name, get_object_vars($this));
    }

    /** ** RELATIONSHIPS ** */


    /**
     * removeItem
     * removes an item and recalculcates amounts
     *
     * @param mixed $mixed : can be an integer or an item instance
     *                       - if an integer, removes the item with
     *                         that position in the collection
     *                       - if an instance, removes that item
     * @author JoeZ99 <jzarate@gmail.com>
     */
    public function removeItem($mixed)
    {
        if ($mixed instanceof ItemRetencion) {
            $items = $this->getItems();
            foreach ($items as $ref => $item) {
                if ($item === $mixed) {
                    unset($items[$ref]);
                    break;
                }
            }
        } elseif (is_int($mixed)) {
            unset($this->items[$mixed]);
        }

    }

    /* ** OTHER ** */

    private $decimals = null;

    public function getRoundedAmount($concept = 'gross')
    {
        if (!in_array($concept, ['base', 'discount', 'net', 'tax', 'gross'])) {
            return 0;
        }

        return round(call_user_func([$this, Inflector::camelize('get_' . $concept . '_amount')]), $this->getDecimals());
    }

    private function getDecimals()
    {
        if (!$this->decimals) {
            $this->decimals = 2;
        }

        return $this->decimals;
    }


    /**
     * calculate values over items
     *
     * Warning!! this method only works when called from a real entity, not
     * the abstract.
     *
     * @param string $field
     * @param boolean $rounded
     * @return float
     */
    public function calculate($field, $rounded = false)
    {
        $val = 0;
        switch ($field) {
            default:
                foreach ($this->getItems() as $item) {
                    $method = Inflector::camelize('get_' . $field);
                    $val += $item->$method();
                }
                break;
        }

        if ($rounded) {
            return round($val, $this->getDecimals());
        }

        return $val;
    }

    public function checkAmounts()
    {
        //$this->setBaseAmount($this->calculate('base'));
        $retenido = $this->calculate('retenido');
        $rounded_gross = round(
            $retenido,
            $this->getDecimals()
        );
        $this->setGrossAmount($rounded_gross);

        return $this;
    }


    /** *********** LIFECYCLE CALLBACKS ************* */

    /**
     * @ORM\PreUpdate
     * @ORM\PrePersist
     */
    public function preSave(LifecycleEventArgs $args)
    {
        $this->checkStatus();
        $providers = $args->getObjectManager()->getRepository(Provider::class)->findBy([
            'name' => $this->getProviderName(),
            'identification' => $this->getProviderIdentification(),
            'empresa' => $this->getEmpresa()->getId(),
        ]);
        $provider = reset($providers);
        if ($provider) {
            $this->setProvider($provider);
        }

        if($this->id === null)
            $this->slug = md5('sv_'.$this->empresa->getId().uniqid());

    }
}
