<?php

namespace App\Entity\Model;

use App\Util\ValidarIdentificacion;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * Customer
 *
 * @ORM\Table(name="customer", uniqueConstraints={@ORM\UniqueConstraint(name="customer_unique", columns={
 *     "identification",
 *     "empresa_id"})}
 *
 *     )
 * @ORM\Entity(repositoryClass="App\Repository\CustomerRepository")
 * @ORM\HasLifecycleCallbacks()
 */
class Customer
{
    /**
     *
     * @ORM\Column(name="id", type="integer")
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="AUTO")
     */
    private  $id = null;

    /**
     * @ORM\Column(name="slug", type="string", length=32, nullable=true, unique=true)
     */
    private  $slug = null;

    /**
     *
     * @ORM\Column(name="name", type="string", length=191)
     * @Assert\NotBlank(message = "Ingrese el nombre")
     */
    private  $name = null;

    /**
     *
     * @ORM\Column(name="identification", type="string", length=128, nullable=true)
     * @Assert\NotBlank(message = "Ingrese en número de identificación")
     */
    private  $identification = null;

    /**
     *
     * @ORM\Column(name="email", type="string", length=191)
     * @Assert\NotBlank(message = "Ingrese el correo electrónico")
     * @Assert\Email()
     */
    private  $email = null;

    /**
     * @ORM\Column(name="contact_person", type="string", length=255, nullable=true)
     */
    private  $contactPerson = null;

    /**
     *
     * @ORM\Column(name="invoicing_address", type="string", length=255)
     * @Assert\NotBlank(message = "Ingrese la dirección")
     */
    private  $invoicingAddress = null;

    /**
     * @ORM\Column(name="shipping_address", type="string",length=255, nullable=true)
     */
    private  $shippingAddress = null;

    /**
     * @ORM\OneToMany(targetEntity="App\Entity\Model\Invoice", mappedBy="customer")
     */
    private $invoices;

    /**
     *
     * @ORM\Column(type="string", length=2, nullable=true)
     * @Assert\NotBlank(message = "Seleccione el tipo de identificación")
     */
    private  $tipoDocumento = null;

    /**
     * @ORM\Column(type="string", length=50, nullable=true)
     */
    private  $telefono = null;

    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Empresa")
     * @ORM\JoinColumn(name="empresa_id", referencedColumnName="id", onDelete="SET NULL")
     * @Assert\NotBlank(allowNull=false, message="empresa no puede ser nulo, corregir en codigo fuente")
     */
    private $empresa;

    /**
     * @ORM\PrePersist
     */
    public function setSlug()
    {
        $this->slug = md5('sv_' . $this->empresa->getId() . $this->empresa->getRuc() . $this->identification);
    }

    /**
     * Get id
     *
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set name
     *
     * @param string $name
     *
     * @return Customer
     */
    public function setName($name)
    {
        $this->name = strtoupper($name);

        return $this;
    }

    /**
     * Get name
     *
     * @return string
     */
    public function getName()
    {
        return $this->name;
    }

    /**
     * Set identification
     *
     * @param string $identification
     *
     * @return Customer
     */
    public function setIdentification($identification)
    {
        $this->identification = $identification;

        return $this;
    }

    /**
     * Get identification
     *
     * @return string
     */
    public function getIdentification()
    {
        return $this->identification;
    }

    /**
     * Set email
     *
     * @param string $email
     *
     * @return Customer
     */
    public function setEmail($email)
    {
        $this->email = $email;

        return $this;
    }

    /**
     * Get email
     *
     * @return string
     */
    public function getEmail()
    {
        return $this->email;
    }

    /**
     * Set contactPerson
     *
     * @param string $contactPerson
     *
     * @return Customer
     */
    public function setContactPerson($contactPerson)
    {
        $this->contactPerson = $contactPerson;

        return $this;
    }

    /**
     * Get contactPerson
     *
     * @return string
     */
    public function getContactPerson()
    {
        return $this->contactPerson;
    }

    /**
     * Set invoicingAddress
     *
     * @param string $invoicingAddress
     *
     * @return Customer
     */
    public function setInvoicingAddress($invoicingAddress)
    {
        $this->invoicingAddress = strtoupper($invoicingAddress);

        return $this;
    }

    /**
     * Get invoicingAddress
     *
     * @return string
     */
    public function getInvoicingAddress()
    {
        return $this->invoicingAddress;
    }

    /**
     * Set shippingAddress
     *
     * @param string $shippingAddress
     *
     * @return Customer
     */
    public function setShippingAddress($shippingAddress)
    {
        $this->shippingAddress = $shippingAddress;

        return $this;
    }

    /**
     * Get shippingAddress
     *
     * @return string
     */
    public function getShippingAddress()
    {
        return $this->shippingAddress;
    }

    public function jsonSerialize()
    {
        return ['id' => $this->getId(), 'name' => $this->getName(), 'email' => $this->getEmail(), 'identification' => $this->getIdentification(), 'contact_person' => $this->getContactPerson(), 'invoicing_address' => $this->getInvoicingAddress(), 'shipping_address' => $this->getShippingAddress()];
    }

    public function label()
    {
        return $this->getName();
    }

    /**
     * @return string
     */
    public function getTipoDocumento()
    {
        return $this->tipoDocumento;
    }

    /**
     * @param string $tipoDocumento
     */
    public function setTipoDocumento($tipoDocumento)
    {
        $this->tipoDocumento = $tipoDocumento;
    }


    /**
     * @return string
     */
    public function getTelefono()
    {
        return $this->telefono;
    }

    /**
     * @param string $telefono
     */
    public function setTelefono($telefono)
    {
        $this->telefono = $telefono;
    }

    /**
     * @return mixed
     */
    public function getEmpresa()
    {
        return $this->empresa;
    }

    /**
     * @param mixed $empresa
     */
    public function setEmpresa($empresa)
    {
        $this->empresa = $empresa;
    }

    /**
     * @return string
     */
    public function getSlug(): string
    {
        return $this->slug;
    }

    public function validarRuc($numero, $tipoDoc)
    {

        $resp = null;
        try {

            if ($tipoDoc == "C" && strlen($numero) != 10)
                return "Número de cédula debe contener 10 dígitos";

            if ($tipoDoc == "R" && strlen($numero) != 13)
                return "Número de RUC debe contener 13 dígitos";

            if ($tipoDoc == "P")
                return null;

            // Crear nuevo objecto
            $validador = new ValidarIdentificacion();

            if (strlen($numero) == 10) {
                // validar CI
                try {
                    if ($validador->validarCedula($numero)) {
                        return null;
                    } else {
                        $resp = $validador->getError();
                    }
                } catch (\Exception $exception) {
                    $resp = 'Cédula incorrecta: ' . $exception->getMessage();
                }
            } elseif (strlen($numero) == 13) {
                try {
                    // validar RUC persona natural
                    if ($validador->validarRucPersonaNatural($numero)) {
                        return $resp;
                    } elseif ($validador->validarRucSociedadPrivada($numero)) {
                        return $resp;
                    } elseif ($validador->validarRucSociedadPublica($numero)) {
                        return $resp;
                    } else {
                        $resp = 'RUC incorrecto: ' . $validador->getError();
                    }
                } catch (\Exception $exception) {
                    $resp = $exception->getMessage();
                }
            }
        } catch (\Exception $exception) {
            $resp = $exception->getMessage();
        }

        return $resp;

    }

}
