<?php

namespace App\Entity\Model;

use App\Util\Inflector;
use Doctrine\ORM\Event\LifecycleEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\Common\Collections\ArrayCollection;
use Symfony\Component\Validator\Constraints as Assert;

/**
 * App\Entity\Model\Ncredito
 *
 * @ORM\Table(indexes={
 *    @ORM\Index(name="ncredito_cstnm_idx", columns={"customer_name"}),
 *    @ORM\Index(name="ncredito_cstid_idx", columns={"customer_identification"}),
 *    @ORM\Index(name="ncredito_cstml_idx", columns={"customer_email"})
 * })
 * @ORM\Entity(repositoryClass="App\Repository\NcreditoRepository")
 * @ORM\HasLifecycleCallbacks()
 */
class Ncredito extends AbstractInvoice
{
    /**
     * @ORM\ManyToMany(targetEntity="Payment", orphanRemoval=true, cascade={"all"})
     * @ORM\JoinTable(name="ncreditos_payments",
     *      joinColumns={@ORM\JoinColumn(name="ncredito_id", referencedColumnName="id", onDelete="CASCADE")},
     *      inverseJoinColumns={@ORM\JoinColumn(
     *          name="payment_id", referencedColumnName="id", unique=true, onDelete="CASCADE"
     *      )}
     * )
     */
    private $payments;

    /**
     * @ORM\Column(name="sent_by_email", type="boolean", nullable=true)
     */
    private  $sent_by_email;

    /**
     * @ORM\Column(name="number", type="integer", nullable=true)
     */
    private  $number = null;

    /**
     * @var date $issue_date
     *
     * @ORM\Column(name="issue_date", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $issue_date;

    /**
     * @var date $due_date
     *
     * @ORM\Column(name="due_date", type="date", nullable=true)
     * @Assert\Type("\DateTime")
     */
    private $due_date;

    /**
     * @ORM\ManyToMany(targetEntity="App\Entity\Model\Item", cascade={"all"}, inversedBy="invoice")
     * @ORM\JoinTable(name="ncreditos_items",
     *      joinColumns={@ORM\JoinColumn(name="ncredito_id", referencedColumnName="id", onDelete="CASCADE")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="item_id", referencedColumnName="id", unique=true, onDelete="CASCADE")}
     * )
     * @Assert\NotBlank()
     */
    public $items;

    /**
     * @ORM\ManyToMany(targetEntity="App\Entity\Model\ItemPago", cascade={"all"}, inversedBy="invoices")
     * @ORM\JoinTable(name="ncreditos_pagos",
     *      joinColumns={@ORM\JoinColumn(name="ncredito_id", referencedColumnName="id", onDelete="CASCADE")},
     *      inverseJoinColumns={@ORM\JoinColumn(name="itempago_id", referencedColumnName="id", unique=true, onDelete="CASCADE")}
     * )
     * @Assert\NotBlank(allowNull=true)
     */
    public $pagos;

    /**
     * @var boolean $closed
     *
     * @ORM\Column(name="closed", type="boolean", nullable=true)
     */
    private $forcefully_closed = false;

    /**
     * @ORM\Column(name="baseiva", type="decimal", scale=2, precision=15, nullable=true)
     */
    private  $baseiva = null;

    /**
     * @ORM\Column(name="basecero", type="decimal", scale=2, precision=15, nullable=true)
     */
    private  $basecero = null;

    /**
     * @ORM\Column(name="claveacceso", type="string", length=49)
     */
    private  $claveAcceso = '';

    /**
     * @ORM\Column(type="string", length=6)
     */
    private  $serie = '';

    /**
     * @var
     *
     * @ORM\Column(type="decimal", scale=2, precision=15, nullable=true)
     */
    private  $porIva = 0.00;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $xml;

    /**
     * @ORM\Column(type="integer", nullable=true)
     */
    private  $ambiente;


    /**
     * @ORM\Column( type="boolean")
     */
    private  $autorizado = false;


    /**
     * @ORM\Column(type="string", length=30,  nullable=true)
     */
    private  $fecha_autorizacion = null;

    /**
     * @ORM\Column( type="boolean")
     */
    private  $sinrespuesta = false;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $mensajeError;

    /**
     * @var text
     *
     * @ORM\Column(type="text", nullable=true)
     */
    private $xmlAutorizado;

    /**
     * @ORM\Column(type="blob", nullable=true)
     */
    private $docpdf;

    /**
     * @ORM\Column( type="boolean")
     */
    private  $anulado = false;

    /**
     * @ORM\Column(type="string", length=6, nullable=true)
     * @Assert\NotBlank(allowNull=true)
     */
    private  $serieDocSustento = null;

    /**
     * @var string
     * @ORM\Column(type="string", length=9, nullable=true)
     * @Assert\NotBlank(allowNull=true)
     */
    private $numeroDocSustento;


    /**
     * @var date $fecDocSustento
     *
     * @ORM\Column(type="date", nullable=true)
     * @Assert\Type("\DateTime")
     * @Assert\NotBlank(allowNull=true)
     */
    private $fecDocSustento;

    /**
     *
     * @ORM\Column(type="string", length=200,  nullable=true)
     * @Assert\NotBlank(allowNull=true)
     */
    private  $motivo = null;


    /**
     * @ORM\ManyToOne(targetEntity="App\Entity\Model\Empresa")
     * @ORM\JoinColumn(name="empresa_id", referencedColumnName="id", onDelete="SET NULL")
     */
    public $empresa;

    public function __construct()
    {
        parent::__construct();
        $this->payments = new ArrayCollection();
        $this->issue_date = new \DateTime();
        $this->due_date = new \DateTime();
    }

    /**
     * @return boolean
     */
    public function isClosed()
    {
        return $this->status === Ncredito::CLOSED;
    }

    /**
     * @return boolean
     */
    public function isOpen()
    {
        return in_array($this->status, [Ncredito::OPENED, Ncredito::OVERDUE], true);
    }

    /**
     * @return boolean
     */
    public function isOverdue()
    {
        return $this->status === Ncredito::OVERDUE;
    }

    /**
     * @return boolean
     */
    public function isDraft()
    {
        return $this->status === Ncredito::DRAFT;
    }

    /**
     * Set sent_by_email
     *
     * @param boolean $sentByEmail
     */
    public function setSentByEmail($sentByEmail)
    {
        $this->sent_by_email = $sentByEmail;
    }

    /**
     * Get sent_by_email
     *
     * @return boolean
     */
    public function isSentByEmail(): bool
    {
        return (bool) $this->sent_by_email;
    }

    /**
     * Set number
     *
     * @param integer $number
     */
    public function setNumber($number)
    {
        $this->number = $number;
    }

    /**
     * Get number
     *
     * @return integer
     */
    public function getNumber()
    {

        return $this->number;
    }

    /**
     * Set issue_date
     *
     * @param date $issueDate
     */
    public function setIssueDate($issueDate)
    {
        $this->issue_date = $issueDate instanceof \DateTime ?
        $issueDate: new \DateTime($issueDate);
    }

    /**
     * Get issue_date
     *
     * @return date
     */
    public function getIssueDate()
    {
        return $this->issue_date;
    }

    /**
     * Set due_date
     *
     * @param date $dueDate
     */
    public function setDueDate($dueDate)
    {
        $this->due_date = $dueDate instanceof \DateTime ?
        $dueDate : new \DateTime($dueDate);
    }

    /**
     * Get due_date
     *
     * @return date
     */
    public function getDueDate()
    {
        return $this->due_date;
    }

    /**
     * Add payments
     *
     * @param App\Entity\Model\Payment $payment
     */
    public function addPayment(\App\Entity\Model\Payment $payment)
    {
        $this->payments[] = $payment;
    }

    /**
     * Removes a payment.
     *
     * @param App\Entity\Model\Payment $payment
     */
    public function removePayment(\App\Entity\Model\Payment $payment)
    {
        foreach ($this->getPayments() as $key => $value) {
            if ($value === $payment) {
                unset($this->payments[$key]);
                break;
            }
        }
    }

    /**
     * Get payments
     *
     * @return Doctrine\Common\Collections\Collection
     */
    public function getPayments()
    {
        return $this->payments;
    }

    public function setForcefullyClosed($value)
    {
        $this->forcefully_closed = $value;
    }

    public function isForcefullyClosed()
    {
        return $this->forcefully_closed;
    }

    /** **************** CUSTOM METHODS AND PROPERTIES **************  */

    /**
     * TODO: provide the serie .
     */
    public function __toString()
    {
        return $this->label();
    }

    public function label(string $draftLabel = '[draft]')
    {
        /*$series = $this->getSeries();
        $label = '';
        $label .= $series ? $series->getValue() : '';
        $label .= $this->isDraft() ? $draftLabel : $this->getNumber();
*/

        $label = sprintf("%09d",$this->getNumber());
        return $label;
    }

    public function labelNumDocSustento(string $draftLabel = '[draft]')
    {
        return sprintf("%09d",$this->numeroDocSustento);
    }

    /**
     * @var int
     */
    public const DRAFT    = 0;

    /**
     * @var int
     */
    public const CLOSED   = 1;

    /**
     * @var int
     */
    public const OPENED   = 2;

    /**
     * @var int
     */
    public const OVERDUE  = 3;

    public function getDueAmount()
    {
        if ($this->isDraft()) {
            return null;
        }

        return $this->getGrossAmount() - $this->getPaidAmount();
    }

    /**
     * try to catch custom methods to be used in twig templates
     */
    public function __get($name)
    {
        if (strpos($name, 'tax_amount_') === 0) {
            return $this->calculate($name, true);
        }

        //$method1 = Inflector::camelize("get_{$name}");
        $method = Inflector::camelize(sprintf('get_%s', $name));
        if (method_exists($this, $method)) {
            return $this->$method();
        }

        return false;
    }

    public function __isset($name)
    {
        if (strpos($name, 'tax_amount_') === 0) {
            return true;
        }

        if ($name == 'due_amount') {
            return true;
        }

        if (array_key_exists($name, get_object_vars($this))) {
            return true;
        }

        return parent::__isset($name);
    }

    public function getStatusString()
    {
        switch ($this->status) {
            case Ncredito::DRAFT;
                $status = 'draft';
             break;
            case Ncredito::CLOSED;
                $status = 'closed';
            break;
            case Ncredito::OPENED;
                $status = 'opened';
            break;
            case Ncredito::OVERDUE:
                $status = 'overdue';
                break;
            default:
                $status = 'unknown';
                break;
        }

        return $status;
    }

    /**
     * checkStatus
     * checks and sets the status
     *
     * @return App\Ncredito $this
     */
    public function checkStatus()
    {
        if ($this->status == Ncredito::DRAFT) {
            return $this;
        }

        if ($this->isForcefullyClosed() || $this->getDueAmount() == 0) {
            $this->setStatus(Ncredito::CLOSED);
        } elseif ($this->getDueDate()->getTimestamp() > strtotime(date('Y-m-d'))) {
            $this->setStatus(Ncredito::OPENED);
        } else {
            $this->setStatus(Ncredito::OVERDUE);
        }

        return $this;
    }

    public function checkAmounts()
    {
        parent::checkAmounts();
        $this->setPaidAmount($this->calculate('paid_amount'));

        return $this;
    }

    public function checkNumber(LifecycleEventArgs $args)
    {
        // compute the number of ncredito
        if ((!$this->number && $this->status != self::DRAFT) ||
            ($args instanceof PreUpdateEventArgs && $args->hasChangedField('series') && $this->status != self::DRAFT)
        ) {
            $repo = $args->getObjectManager()->getRepository(Ncredito::class);
            $series = $this->getSeries();
            if ($repo && $series) {
                $this->setNumber($repo->getNextNumber($series, $this->getEmpresa()->getId()));
            }
        }
    }

    /* ********** LIFECYCLE CALLBACKS *********** */

    /**
     * @ORM\PrePersist
     * @ORM\PreUpdate
     */
    public function preSave(LifecycleEventArgs $args)
    {
        $this->setUpdatedAt(new \DateTime('now'));

        if ($this->getCreatedAt() == null) {
            $this->setCreatedAt(new \DateTime('now'));
        }

        $this->checkAmounts();
        parent::presave($args);

        $this->checkNumber($args);

    }

    /**
     * @return string
     */
    public function getCustomermicro()
    {
        return $this->customermicro;
    }

    /**
     * @param string $customermicro
     */
    public function setCustomermicro(string $customermicro)
    {
        $this->customermicro = $customermicro;
    }

    /**
     * @return decimal
     */
    public function getBaseiva()
    {
        return $this->baseiva;
    }

    /**
     * @param decimal $baseiva
     */
    public function setBaseiva($baseiva)
    {
        $this->baseiva = $baseiva;
    }

    /**
     * @return decimal
     */
    public function getBasecero()
    {
        return $this->basecero;
    }

    /**
     * @param decimal $basecero
     */
    public function setBasecero($basecero)
    {
        $this->basecero = $basecero;
    }

    /**
     * @return mixed
     */
    public function getPorIva()
    {
        return $this->porIva;
    }

    /**
     * @param mixed $porIva
     */
    public function setPorIva($porIva)
    {
        $this->porIva = $porIva;
    }

    /**
     * @return string
     */
    public function getClaveAcceso()
    {
        return $this->claveAcceso;
    }

    /**
     * @param string $claveAcceso
     */
    public function setClaveAcceso($claveAcceso)
    {
        $this->claveAcceso = $claveAcceso;
    }

    /**
     * @return string
     */
    public function getSerie()
    {
        return $this->serie;
    }

    /**
     * @param string $serie
     */
    public function setSerie($serie)
    {
        $this->serie = $serie;
    }

    /**
     * @return bool
     */
    public function getAutorizado()
    {
        return $this->autorizado;
    }

    /**
     * @param bool $autorizado
     */
    public function setAutorizado($autorizado)
    {
        $this->autorizado = $autorizado;
    }

    /**
     * @return string
     */
    public function getAutorizacion()
    {
        return $this->autorizacion;
    }

    /**
     * @param string $autorizacion
     */
    public function setAutorizacion($autorizacion)
    {
        $this->autorizacion = $autorizacion;
    }


    /**
     * @return text
     */
    public function getXml()
    {
        return $this->xml;
    }

    /**
     * @param text $xml
     */
    public function setXml($xml)
    {
        $this->xml = $xml;
    }

    /**
     * @return mixed
     */
    public function getEmpresa()
    {
        return $this->empresa;
    }

    /**
     * @param mixed $empresa
     */
    public function setEmpresa($empresa)
    {
        $this->empresa = $empresa;
    }

    /**
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getPagos()
    {
        return $this->pagos;
    }

    /**
     * @param mixed $pagos
     */
    public function setPagos($pagos)
    {
        $this->pagos = $pagos;
    }

    /**
     * @return string
     */
    public function getFechaAutorizacion()
    {
        return $this->fecha_autorizacion;
    }

    /**
     * @param string $fecha_autorizacion
     */
    public function setFechaAutorizacion(string $fecha_autorizacion)
    {
        $this->fecha_autorizacion = $fecha_autorizacion;
    }

    /**
     * @return int
     */
    public function getAmbiente()
    {
        return $this->ambiente;
    }

    /**
     * @param int $ambiente
     */
    public function setAmbiente($ambiente)
    {
        $this->ambiente = $ambiente;
    }

    /**
     * @param mixed $ambiente
     */
    public function getLabelAmbiente()
    {
        if ($this->ambiente === 1) {
            return "PRUEBAS";
        } elseif ($this->ambiente === 2) {
            return "PRODUCCION";
        } else
            return "UNDEFINED";

    }


    /**
     * @return bool
     */
    public function isSinrespuesta()
    {
        return $this->sinrespuesta;
    }

    /**
     * @param bool $sinrespuesta
     */
    public function setSinrespuesta( $sinrespuesta)
    {
        $this->sinrespuesta = $sinrespuesta;
    }

    /**
     * @return text
     */
    public function getMensajeError()
    {
        return $this->mensajeError;
    }

    /**
     * @param text $mensajeError
     */
    public function setMensajeError($mensajeError)
    {
        $this->mensajeError = $mensajeError;
    }

    /**
     * @return text
     */
    public function getXmlAutorizado()
    {
        return $this->xmlAutorizado;
    }

    /**
     * @param text $xmlAutorizado
     */
    public function setXmlAutorizado($xmlAutorizado)
    {
        $this->xmlAutorizado = $xmlAutorizado;
    }

    /**
     * @return mixed
     */
    public function getDocpdf()
    {
        return $this->docpdf;
    }

    /**
     * @param mixed $docpdf
     */
    public function setDocpdf($docpdf)
    {
        $this->docpdf = $docpdf;
    }

    /**
     * @return bool
     */
    public function getAnulado(): bool
    {
        return $this->anulado;
    }

    /**
     * @param bool $anulado
     */
    public function setAnulado(bool $anulado): void
    {
        $this->anulado = $anulado;
    }

    /**
     * @return string
     */
    public function getSerieDocSustento()
    {
        return $this->serieDocSustento;
    }

    /**
     * @param string $serieDocSustento
     */
    public function setSerieDocSustento(string $serieDocSustento): void
    {
        $this->serieDocSustento = $serieDocSustento;
    }

    /**
     * @return string
     */
    public function getNumeroDocSustento()
    {
        return $this->numeroDocSustento;
    }

    /**
     * @param string $numeroDocSustento
     */
    public function setNumeroDocSustento(string $numeroDocSustento): void
    {
        $this->numeroDocSustento = sprintf("%09d", $numeroDocSustento);
    }


    /**
     * @return date
     */
    public function getFecDocSustento()
    {
        return $this->fecDocSustento;
    }

    /**
     * @param date $fecDocSustento
     */
    public function setFecDocSustento($fecDocSustento): void
    {
        $this->fecDocSustento = $fecDocSustento;
    }

    /**
     * @return string
     */
    public function getMotivo()
    {
        return $this->motivo;
    }

    /**
     * @param string $motivo
     */
    public function setMotivo(string $motivo): void
    {
        $this->motivo = strtoupper($motivo);
    }

    /**
     * @return string
     */
    public function getEstado(): string
    {
        if ($this->autorizado)
            return 'A';
        else
            return 'N';
    }

}
