<?php

namespace App\Controller\Base;

use App\Electronico\Comprobante;
use App\Entity\Model\AbstractInvoice;
use App\Entity\Model\Empresa;
use App\Entity\Model\Invoice;
use App\Entity\Model\Item;
use App\Entity\Model\ItemInvoiceReembolso;
use App\Entity\Model\Provider;
use App\Entity\Model\Tax;
use App\Util\RestApiFunciones;
use Doctrine\ORM\EntityManagerInterface;
use JMS\Serializer\SerializerBuilder;
use Symfony\Component\Config\Definition\Exception\Exception;
use Symfony\Component\Form\Extension\Core\DataTransformer\MoneyToLocalizedStringTransformer;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;

abstract class AbstractInvoiceController extends AbstractController
{

    public $translator;

    protected function cargarImpuestos(AbstractInvoice $invoice)
    {
        $basecero = 0;
        $baseiva = 0;
        $valiva=0;

        foreach ($invoice->getItems() as $item) {

            $iva = 0;
            $p = $item->getProduct();

            //$item = new Item();
            if($item->getDiscount()*1 >= 100)
                $item->setDiscount($item->getDiscountPercent());

            $item->setNeto($item->getNetAmount());
            $item->setValdescuento($item->getDiscountAmount());

            $subtotal = $item->getQuantity() * $item->getUnitaryCost();

            if ($item->getTaxes() != null) {

                $tax = $item->getTaxes();
                //foreach ($postItem['taxes'] as $taxId) {
                //$tax = $taxRepo->find($taxId);
                //$tax = new Tax();
                if ($tax != null) {
                    $porcentaje = (double)$tax->getValue();
                    if ($porcentaje > 0) {
                        //continue;
                        $iva = $item->getNeto() * ($porcentaje / 100);

                        //$baseiva += $subtotal;
                        $baseiva += $item->getNeto();
                        $valiva += $iva;

                        $item->setValiva($iva);
                        $item->addTax($tax);
                        $item->setSubtotal($subtotal);
                        $invoice->setPorIva($tax->getValue());
                    } else {
                        $basecero += $item->getNeto();

                        $item->setValiva(0);
                        $item->addTax($tax);
                        $item->setSubtotal($subtotal);
                        $invoice->setPorIva($tax->getValue());
                    }
                }
            }


            //$item->addTax($tax);
            //}
            /*} else {
                $basecero += $subtotal;

                $item->setSubtotal($subtotal);
                $item->setValIva($iva);
            }
            */
        }

        $invoice->setTotalFactura($basecero+$baseiva+$valiva);
        $invoice->setBasecero($basecero);
        $invoice->setBaseiva($baseiva);
    }

    protected function cargarReembolsos(AbstractInvoice $invoice, $em)
    {
        $baseImponible = 0;
        $valorIva = 0;

        foreach ($invoice->getReembolsos() as $reembolso) {

            //$reembolso = new Reembolso();
            $providers = $em->getRepository(Provider::class)->findBy([
                'name' => $reembolso->getProvreembolsoName(),
                'identification' => $reembolso->getProvreembolsoIdentification(),
            ]);
            $provider = reset($providers);
            if ($provider) {
                $reembolso->setProvider($provider);
            }

            $reembolso->setInvoice($invoice);
            $subtotal = $reembolso->getBaseAmount();

            if ($reembolso->getTaxes() != null) {

                $tax = $reembolso->getTaxes();
                //foreach ($postItem['taxes'] as $taxId) {
                //$tax = $taxRepo->find($taxId);
                //$tax = new Tax();
                if ($tax != null) {
                    $porcentaje = (double)$tax->getValue();
                    if ($porcentaje > 0) {
                        //continue;
                        $iva = $subtotal * ($porcentaje / 100);

                        $valorIva += $iva;

                        $reembolso->setValiva($iva);
                        $reembolso->addTax($tax);
                        $reembolso->setImponible($subtotal);
                        $reembolso->setPorIva($tax->getValue());
                        $reembolso->setBaseIva($subtotal);
                    } else {

                        $reembolso->setValiva(0);
                        $reembolso->addTax($tax);
                        $reembolso->setImponible($subtotal);
                        $reembolso->setPorIva($tax->getValue());
                        $reembolso->setBaseCero($subtotal);
                    }
                }
            }

            $baseImponible += $subtotal;
        }

        $invoice->setBaseReemboAmount($baseImponible);
        $invoice->setTaxReemboAmount($valorIva);
        $invoice->setGrossReemboAmount($baseImponible+$valorIva);
    }

    protected function consultarAutorizacion(Invoice $invoice, EntityManagerInterface $entityManager)
    {
        $respuesta = null;
        try {
            $em = $entityManager;

            $app_url = $this->getParameter('api_url');

            $error = false;

            $resp = RestApiFunciones::consultarAutoComprobante($error, $app_url, $invoice->getClaveAcceso());

            if ($error) {
                $this->addTranslatedMessage($resp, 'danger');
                $invoice->setSinrespuesta(true);
                $invoice->setMensajeError($resp);
                $em->persist($invoice);
                $em->flush();
            } elseif ($resp->autorizado) {
                $this->addTranslatedMessage('Comprobante autorizado, fecha: ' . $resp->fecha);
                $invoice->setAutorizado(true);
                $invoice->setSinrespuesta(false);
                $invoice->setMensajeError("");
                $invoice->setFechaAutorizacion($resp->fecha);
                $invoice->setXmlAutorizado($resp->comprobante);
                $invoice->setStatus(Invoice::CLOSED);
                $invoice->setForcefullyClosed(true);
                $em->persist($invoice);
                $em->flush();
            } else {
                $this->addTranslatedMessage($resp, 'danger');
            }
        } catch (Exception $exception) {
            return $exception->getMessage();
        }

        return $respuesta;
    }

    public $logger;

    protected function generarXml(Invoice $invoice, EntityManagerInterface $entityManager)
    {
        try {
            $empresa = $invoice->getEmpresa();

            $serie = $invoice->getSerie();

            $numero = str_pad($invoice->getNumber(), 9, "0", STR_PAD_LEFT);

            $emisor = new Comprobante(null, $empresa, $serie);

            $resp = $emisor->generaClave($invoice->getIssueDate()->format('d/m/Y'), "01", $numero);

            if ($resp !== null) {
                $this->addTranslatedMessage($resp, 'danger');
                return null;
            }

            $invoice->setClaveAcceso($emisor->getClaveAcceso());

            $error = false;
            $factura = $emisor->generarXml($invoice, $error);

            if ($error)
                $this->addTranslatedMessage($factura, 'danger');

            try {
                $serializer = SerializerBuilder::create()->build();
                $xml = $serializer->serialize($factura, 'xml');
                $xml = trim(preg_replace('#\s+#', ' ', $xml));
                $invoice->setXml($xml);
            } catch (Exception $exception) {
                $this->addTranslatedMessage($exception->getMessage(), 'danger');
                return null;
            }

            $em = $entityManager;
            $em->persist($invoice);
            $em->flush();

        } catch (Exception $exception) {
            $this->addTranslatedMessage($exception->getMessage(), 'danger');
            return null;
        }

        return $invoice;
    }

    protected function getInvoiceReembolsoTotalsFromPost(array $post, AbstractInvoice $invoice, string $locale, EntityManagerInterface $entityManager): array
    {
        $user = $this->getUser();

        $em = $entityManager;

        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $taxRepo = $em->getRepository(Tax::class);
        //$currency = $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR');
        $currency = $empresa == null ? 'USD' : $empresa->getCurrency();
        $formatter = new \NumberFormatter($locale, \NumberFormatter::CURRENCY);
        $transformer = new MoneyToLocalizedStringTransformer(2, true);

        $totals = [];
        foreach ($post['items'] as $index => $postItem) {
            $item = new Item($taxRepo->findTaxDefault($empresa->getId()));
            $item->setUnitaryCost($transformer->reverseTransform($postItem['unitary_cost']));
            $item->setQuantity($postItem['quantity']);
            $item->setDiscount($postItem['discount_percent']);
            if (isset($postItem['taxes'])) {
                //foreach($postItem['taxes'] as $taxId) {
                $tax = $taxRepo->find($postItem['taxes']);
                if (!$tax instanceof \App\Entity\Model\Tax) {
                    continue;
                }

                $item->setTaxes($tax);
                //}

                //$item->setTaxes($postItem['taxes']);
            }

            $totals['items'][$index] = [
                'gross_amount' => $formatter->formatCurrency($item->getGrossAmount(), $currency),
            ];
            $invoice->addItem($item);
        }

        $invoice->checkAmounts();

        $pagoTotal = 0;
        foreach ($post['pagos'] as $index => $pago) {
            $pagoTotal += $transformer->reverseTransform($pago['valor']);
        }

        return $totals + [
                'liquidacion_base_amount' => $formatter->formatCurrency($invoice->getBaseAmount(), $currency),
                'liquidacion_tax_amount' => $formatter->formatCurrency($invoice->getTaxAmount(), $currency),
                'liquidacion_gross_amount' => $formatter->formatCurrency($invoice->getGrossAmount(), $currency),
                'pago_gross_amount' => number_format($invoice->getGrossAmount(), 2),
            ];
    }

    protected function getInvoiceTotalsFromPost(array $post, AbstractInvoice $invoice, string $locale, EntityManagerInterface $entityManager): array
    {
        $user = $this->getUser();

        $em = $entityManager;

        $empresaRepo = $em->getRepository(Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $taxRepo = $em->getRepository(Tax::class);
        //$currency = $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR');
        $currency = $empresa == null ? 'USD' : $empresa->getCurrency();
        $formatter = new \NumberFormatter($locale, \NumberFormatter::CURRENCY);
        //$transformer = new MoneyToLocalizedStringTransformer(2, true);
        $transformer = new MoneyToLocalizedStringTransformer($empresa->getDecPunit(), true);

        $totals = [];
        foreach ($post['items'] as $index => $postItem) {
            $item = new Item($taxRepo->findTaxDefault($empresa->getId()));
            $item->setUnitaryCost($transformer->reverseTransform($postItem['unitary_cost']));
            $item->setQuantity($postItem['quantity']);
            $item->setDiscount($postItem['discount_percent']);
            if (isset($postItem['taxes'])) {
                //foreach($postItem['taxes'] as $taxId) {
                $tax = $taxRepo->find($postItem['taxes']);
                if (!$tax instanceof \App\Entity\Model\Tax) {
                    continue;
                }

                $item->setTaxes($tax);
                //}

                //$item->setTaxes($postItem['taxes']);
            }

            $totals['items'][$index] = [
                'gross_amount' => $formatter->formatCurrency($item->getGrossAmount(), $currency),
            ];
            $invoice->addItem($item);
        }

        $invoice->checkAmounts();

        $pagoTotal = 0;
        foreach ($post['pagos'] as $index => $pago) {
            $pagoTotal += $transformer->reverseTransform($pago['valor']);
        }

        return $totals + [
            'invoice_base_amount' => $formatter->formatCurrency($invoice->getBaseAmount(), $currency),
            'invoice_tax_amount' => $formatter->formatCurrency($invoice->getTaxAmount(), $currency),
            'invoice_gross_amount' => $formatter->formatCurrency($invoice->getGrossAmount(), $currency),
            'pago_gross_amount' => number_format($invoice->getGrossAmount(), 2),
        ];
    }

    protected function addTranslatedMessage($message, $status = 'success')
    {
        $this->addFlash($status, $this->translator->trans($message, [], 'invoice'));
    }

    protected function bulkDelete(array $invoices, EntityManagerInterface $entityManager)
    {
        $this->denyAccessUnlessGranted('IS_AUTHENTICATED_FULLY');

        $em = $entityManager;

        foreach ($invoices as $invoice) {
            $em->remove($invoice);
        }

        $em->flush();
        $this->addTranslatedMessage('flash.bulk_deleted');

        return $this->redirectToRoute('invoice_index');
    }

    protected function bulkPdf(array $invoices)
    {
        $pages = [];
        foreach ($invoices as $invoice) {
            $pages[] = $this->getInvoicePrintPdfHtml($invoice);
        }

        $html = $this->get('siwapp_core.html_page_merger')->merge($pages, '<div class="pagebreak"> </div>');
        $pdf = $this->getPdf($html);

        return new Response($pdf, \Symfony\Component\HttpFoundation\Response::HTTP_OK, [
            'Content-Type' => 'application/pdf',
            'Content-Disposition' => 'attachment; filename="Invoices.pdf"'
        ]);
    }

    protected function bulkPrint(array $invoices)
    {
        $pages = [];
        foreach ($invoices as $invoice) {
            $pages[] = $this->getInvoicePrintPdfHtml($invoice, true);
        }

        $html = $this->get('siwapp_core.html_page_merger')->merge($pages, '<div class="pagebreak"> </div>');

        return new Response($html);
    }

    protected function bulkEmail(array $invoices, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;
        foreach ($invoices as $invoice) {
            $message = $this->getEmailMessage($invoice);
            $result = $this->get('mailer')->send($message);
            if ($result) {
                $invoice->setSentByEmail(true);
                $em->persist($invoice);
            }
        }

        $em->flush();
        $this->addTranslatedMessage('flash.bulk_emailed');

        return $this->redirectToRoute('invoice_index');
    }

    protected function generarPdf(Invoice $invoice)
    {
        $filename = 'FAC_' . $invoice->getSerie() . "-" . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT) . '.pdf';

        $xml = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url').'facturaride.php';

        $error = false;
        $mensaje = "";
        $empresa = $invoice->getEmpresa();
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $xml, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            file_put_contents($filename, $docPdf);

            header("Cache-Control: public");
            header("Content-Description: File Transfer");
            header(sprintf('Content-Disposition: attachment; filename=%s', $filename));
            header("Content-Type: application/pdf");
            header('Content-Length: ' . filesize($filename));
            //header("Content-Transfer-Encoding: binary");
            header('Accept-Ranges: bytes');
            echo $docPdf;

            try {
                unlink($filename);
            } catch (\Exception $exception) {

            }

        }
    }

    protected function delete(Invoice $invoice, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;

        if ($invoice->getAutorizado() && $invoice->getAmbiente() == 2 ) {
            $this->addTranslatedMessage('Factura no puede ser eliminada, estado: AUTORIZADO, ambiente: PRODUCCION', 'warning');
            return false;
        } else {
            $em->remove($invoice);
            $em->flush();
            $this->addTranslatedMessage('flash.deleted');

            return true;
        }
    }

    protected function enviarMail($email, Invoice $invoice, EntityManagerInterface $entityManager)
    {
        $em = $entityManager;

        $xmlAutorizado = null;

        if ($invoice->getAutorizado())
            $xmlAutorizado = $invoice->getXmlAutorizado();

        $numero = $invoice->getSerie() . '-' . str_pad($invoice->getNumber(), 9, '0', STR_PAD_LEFT);

        $error = false;
        $mensaje = "";

        $data = $invoice->getAutorizado() ? $invoice->getXmlAutorizado() : $invoice->getXml();

        $app_url = $this->getParameter('api_url');

        $app_url .= 'facturaride.php';
        $empresa = $invoice->getEmpresa();
        $logo = null;
        if($empresa->getRutaLogo()) {
            if (strlen(trim($empresa->getRutaLogo())) > 0) {
                $logoaux = trim($empresa->getRutaLogo());
                $logoaux = explode("/", $logoaux);
                if(sizeof($logoaux) === 2){
                    $logo = $logoaux[1];
                }
            }
        }
        $docPdf = RestApiFunciones::getPdf($error, $app_url, $invoice->getClaveAcceso(), $data, $mensaje, $logo);

        if ($error) {
            $this->addTranslatedMessage('ERRROR AL GENERAR EL PDF, ' . $mensaje, 'danger');
        } else {
            $result = RestApiFunciones::envioMailComprobante(
                $invoice->getCustomerName(),
                $numero,
                $invoice->getClaveAcceso(),
                $invoice->getFechaAutorizacion(),
                "Factura",
                $email,
                $docPdf,
                $xmlAutorizado,
                $invoice->getEmpresa()
            );

            if ($result == null || $result === null ) {
                $this->addTranslatedMessage('MENSAJE ENVIADO A: ' . $email);
                $invoice->setSentByEmail(true);
                $em->persist($invoice);
                $em->flush();
            } else
                $this->addTranslatedMessage('ERRROR, ' . $result, 'danger');
        }
    }

    protected function getItemsInvoiceReembolsosTotalsFromPost(array $post, AbstractInvoice $invoice, string $locale, EntityManagerInterface $entityManager): array
    {
        $user = $this->getUser();

        $em = $entityManager;

        $empresaRepo = $em->getRepository(\App\Entity\Model\Empresa::class);
        $empresa = $empresaRepo->findOneByUser($user->getRuc());

        $taxRepo = $em->getRepository(\App\Entity\Model\Tax::class);
        //$currency = $em->getRepository('SiwappConfigBundle:Property')->get('currency', 'EUR');
        $currency = $empresa == null ? 'USD' : $empresa->getCurrency();
        $formatter = new \NumberFormatter($locale, \NumberFormatter::CURRENCY);
        $transformer = new MoneyToLocalizedStringTransformer(2, true);

        $totals = [];
        foreach ($post['reembolsos'] as $index => $postItem) {
            $item = new ItemInvoiceReembolso();
            $item->setImponible($transformer->reverseTransform($postItem['imponible']));
            //$item->setQuantity($postItem['quantity']);
            if (isset($postItem['taxes'])) {
                //foreach($postItem['taxes'] as $taxId) {
                $tax = $taxRepo->find($postItem['taxes']);
                if (!$tax instanceof \App\Entity\Model\Tax) {
                    continue;
                }

                $item->setTaxes($tax);
                //}

                //$item->setTaxes($postItem['taxes']);
            }

            $totals['items'][$index] = [
                'gross_reembo_amount' => $formatter->formatCurrency($item->getImponible(), $currency),
            ];
            $invoice->addReembolso($item);
        }

        $invoice->checkReemboAmounts();

        return $totals + [
                'base_reembo_amount' => $formatter->formatCurrency($invoice->getBaseReemboAmount(), $currency),
                'tax_reembo_amount' => $formatter->formatCurrency($invoice->getTaxReemboAmount(), $currency),
                'gross_reembo_amount' => $formatter->formatCurrency($invoice->getGrossReemboAmount(), $currency),
            ];
    }

}

